<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>MEV Data Pipeline - System Dashboard</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: #0f0f23;
            color: #e0e0e0;
            padding: 20px;
        }
        .container { max-width: 1400px; margin: 0 auto; }
        h1 { color: #00d4ff; margin-bottom: 30px; font-size: 28px; }
        h2 { color: #00d4ff; margin: 20px 0 10px; font-size: 20px; border-bottom: 2px solid #00d4ff; padding-bottom: 5px; }
        
        .grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .card {
            background: #1a1a2e;
            border: 1px solid #2a2a3e;
            border-radius: 8px;
            padding: 20px;
        }
        
        .card h3 {
            color: #00d4ff;
            margin-bottom: 15px;
            font-size: 16px;
        }
        
        .stat {
            display: flex;
            justify-content: space-between;
            padding: 10px 0;
            border-bottom: 1px solid #2a2a3e;
        }
        
        .stat:last-child { border-bottom: none; }
        
        .stat-label { color: #999; }
        .stat-value { color: #00ff88; font-weight: bold; }
        
        .status {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: bold;
        }
        
        .status.healthy { background: #00ff88; color: #000; }
        .status.degraded { background: #ffaa00; color: #000; }
        .status.unhealthy { background: #ff3366; color: #fff; }
        
        table {
            width: 100%;
            border-collapse: collapse;
            margin: 20px 0;
            background: #1a1a2e;
        }
        
        th, td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #2a2a3e;
        }
        
        th {
            background: #16213e;
            color: #00d4ff;
            font-weight: 600;
        }
        
        tr:hover { background: #16213e; }
        
        .refresh-btn {
            background: #00d4ff;
            color: #000;
            border: none;
            padding: 10px 20px;
            border-radius: 5px;
            cursor: pointer;
            font-weight: bold;
            margin-bottom: 20px;
        }
        
        .refresh-btn:hover { background: #00aacc; }
        
        .error { color: #ff3366; }
        .success { color: #00ff88; }
        .warning { color: #ffaa00; }
        
        code {
            background: #16213e;
            padding: 2px 6px;
            border-radius: 3px;
            font-family: 'Courier New', monospace;
            font-size: 12px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>⚡ MEV Data Pipeline - System Dashboard</h1>
        
        <button class="refresh-btn" onclick="location.reload()">🔄 Refresh Dashboard</button>
        
        <?php
        error_reporting(E_ALL);
        ini_set('display_errors', 1);
        
        require_once __DIR__ . '/../config/database.php';
        require_once __DIR__ . '/../helpers/Utils.php';
        
        try {
            $pdo = getDatabaseConnection();
            
            // System Health
            echo "<h2>System Health</h2>";
            echo "<div class='grid'>";
            
            // Database Status
            echo "<div class='card'>";
            echo "<h3>Database Connection</h3>";
            try {
                $stmt = $pdo->query("SELECT VERSION() as version");
                $db = $stmt->fetch();
                echo "<div class='stat'><span class='stat-label'>Status</span><span class='stat-value success'>✓ Connected</span></div>";
                echo "<div class='stat'><span class='stat-label'>Version</span><span class='stat-value'>{$db['version']}</span></div>";
            } catch (Exception $e) {
                echo "<div class='stat'><span class='stat-label'>Status</span><span class='stat-value error'>✗ Error</span></div>";
            }
            echo "</div>";
            
            // API Usage
            echo "<div class='card'>";
            echo "<h3>API Usage (Today)</h3>";
            $stmt = $pdo->query("SELECT COUNT(*) as calls FROM api_call_log WHERE DATE(called_at) = CURDATE()");
            $apiUsage = $stmt->fetch();
            $calls = $apiUsage['calls'];
            $limit = RATE_LIMIT_MAX_DAILY_CALLS;
            $percentage = round(($calls / $limit) * 100, 2);
            
            echo "<div class='stat'><span class='stat-label'>Total Calls</span><span class='stat-value'>" . number_format($calls) . "</span></div>";
            echo "<div class='stat'><span class='stat-label'>Daily Limit</span><span class='stat-value'>" . number_format($limit) . "</span></div>";
            echo "<div class='stat'><span class='stat-label'>Usage</span><span class='stat-value'>$percentage%</span></div>";
            echo "</div>";
            
            // Gas Tracker Status
            echo "<div class='card'>";
            echo "<h3>Gas Tracker</h3>";
            $stmt = $pdo->query("SELECT * FROM gas_tracker ORDER BY timestamp DESC LIMIT 1");
            $gas = $stmt->fetch();
            if ($gas) {
                echo "<div class='stat'><span class='stat-label'>Safe Gas</span><span class='stat-value'>{$gas['safe_gas_price']} gwei</span></div>";
                echo "<div class='stat'><span class='stat-label'>Fast Gas</span><span class='stat-value'>{$gas['fast_gas_price']} gwei</span></div>";
                echo "<div class='stat'><span class='stat-label'>Network Util</span><span class='stat-value'>{$gas['network_utilization']}%</span></div>";
                echo "<div class='stat'><span class='stat-label'>Last Update</span><span class='stat-value'>" . Utils::timeAgo($gas['timestamp']) . "</span></div>";
            } else {
                echo "<div class='stat'><span class='stat-label'>Status</span><span class='stat-value error'>No data</span></div>";
            }
            echo "</div>";
            
            // Opportunities
            echo "<div class='card'>";
            echo "<h3>MEV Opportunities</h3>";
            $stmt = $pdo->query("SELECT COUNT(*) as total FROM mev_opportunities WHERE status = 'pending' AND expires_at > NOW()");
            $oppCount = $stmt->fetch();
            $stmt = $pdo->query("SELECT COUNT(*) as total FROM mev_opportunities");
            $oppTotal = $stmt->fetch();
            
            echo "<div class='stat'><span class='stat-label'>Active</span><span class='stat-value'>{$oppCount['total']}</span></div>";
            echo "<div class='stat'><span class='stat-label'>Total Detected</span><span class='stat-value'>{$oppTotal['total']}</span></div>";
            echo "</div>";
            
            echo "</div>";
            
            // Recent Opportunities
            echo "<h2>Recent MEV Opportunities</h2>";
            $stmt = $pdo->query("
                SELECT opportunity_type, net_profit_usd, confidence_score, urgency_level, detected_at, expires_at, status
                FROM mev_opportunities 
                ORDER BY detected_at DESC 
                LIMIT 10
            ");
            $opportunities = $stmt->fetchAll();
            
            if ($opportunities) {
                echo "<table>";
                echo "<tr><th>Type</th><th>Net Profit</th><th>Confidence</th><th>Urgency</th><th>Status</th><th>Detected</th></tr>";
                foreach ($opportunities as $opp) {
                    $profit = number_format($opp['net_profit_usd'], 2);
                    $conf = number_format($opp['confidence_score'], 2);
                    $urgency = $opp['urgency_level'];
                    $status = $opp['status'];
                    $detected = Utils::timeAgo($opp['detected_at']);
                    
                    echo "<tr>";
                    echo "<td>{$opp['opportunity_type']}</td>";
                    echo "<td>\$$profit</td>";
                    echo "<td>$conf</td>";
                    echo "<td><span class='status " . strtolower($urgency) . "'>$urgency</span></td>";
                    echo "<td>$status</td>";
                    echo "<td>$detected</td>";
                    echo "</tr>";
                }
                echo "</table>";
            } else {
                echo "<p>No opportunities detected yet.</p>";
            }
            
            // Recent Errors
            echo "<h2>Recent Errors (Last 24 Hours)</h2>";
            $stmt = $pdo->query("
                SELECT severity, error_message, created_at 
                FROM error_log 
                WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
                ORDER BY created_at DESC 
                LIMIT 10
            ");
            $errors = $stmt->fetchAll();
            
            if ($errors) {
                echo "<table>";
                echo "<tr><th>Severity</th><th>Message</th><th>Time</th></tr>";
                foreach ($errors as $error) {
                    $severity = $error['severity'];
                    $class = strtolower($severity) === 'critical' || strtolower($severity) === 'error' ? 'error' : 'warning';
                    echo "<tr>";
                    echo "<td><span class='$class'>$severity</span></td>";
                    echo "<td>" . substr($error['error_message'], 0, 100) . "</td>";
                    echo "<td>" . Utils::timeAgo($error['created_at']) . "</td>";
                    echo "</tr>";
                }
                echo "</table>";
            } else {
                echo "<p class='success'>✓ No errors in the last 24 hours!</p>";
            }
            
            // API Endpoints
            echo "<h2>API Endpoints</h2>";
            echo "<div class='card'>";
            echo "<ul style='list-style: none; padding: 0;'>";
            echo "<li style='padding: 8px 0;'><code>GET /api/get_opportunities.php</code> - Get MEV opportunities</li>";
            echo "<li style='padding: 8px 0;'><code>GET /api/get_live_data.php</code> - Get live market data</li>";
            echo "<li style='padding: 8px 0;'><code>GET /api/get_gas_prices.php</code> - Get current gas prices</li>";
            echo "<li style='padding: 8px 0;'><code>GET /api/get_whale_activity.php</code> - Get whale movements</li>";
            echo "</ul>";
            echo "</div>";
            
        } catch (Exception $e) {
            echo "<div class='card error'>";
            echo "<h3>Error</h3>";
            echo "<p>{$e->getMessage()}</p>";
            echo "</div>";
        }
        ?>
        
        <p style="margin-top: 40px; text-align: center; color: #666; font-size: 12px;">
            MEV Data Pipeline v1.0.0 | Last Updated: <?php echo date('Y-m-d H:i:s'); ?>
        </p>
    </div>
</body>
</html>
