<?php
/**
 * File: api/get_live_data.php
 * Purpose: REST API endpoint to fetch real-time market data
 * Access: https://core.poipoe.com/mevdata/api/get_live_data.php
 * Author: MEV Pipeline System
 * Last Modified: 2025-11-15
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// CORS headers
header('Access-Control-Allow-Origin: https://app.poipoe.com');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../helpers/Logger.php';
require_once __DIR__ . '/../helpers/Utils.php';

try {
    if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
        throw new Exception("Only GET requests allowed");
    }
    
    $type = $_GET['type'] ?? 'all';
    $pdo = getDatabaseConnection();
    
    $response = [
        'success' => true,
        'timestamp' => date('Y-m-d H:i:s')
    ];
    
    // Get gas data
    if ($type === 'gas' || $type === 'all') {
        $response['gas_data'] = getGasData($pdo);
    }
    
    // Get whale activity
    if ($type === 'whale_activity' || $type === 'all') {
        $response['whale_activity'] = getWhaleActivity($pdo);
    }
    
    // Get DEX volume
    if ($type === 'dex_volume' || $type === 'all') {
        $response['dex_volume'] = getDexVolume($pdo);
    }
    
    echo json_encode($response, JSON_PRETTY_PRINT);
    
} catch (Exception $e) {
    Logger::error("API error in get_live_data", [
        'error' => $e->getMessage()
    ]);
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Failed to fetch live data'
    ]);
}

function getGasData($pdo) {
    try {
        $stmt = $pdo->query("
            SELECT 
                safe_gas_price,
                propose_gas_price,
                fast_gas_price,
                base_fee,
                network_utilization,
                timestamp
            FROM gas_tracker
            ORDER BY timestamp DESC
            LIMIT 1
        ");
        
        $data = $stmt->fetch();
        
        if (!$data) {
            return null;
        }
        
        $utilization = $data['network_utilization'];
        $networkStatus = 'NORMAL';
        if ($utilization > 80) {
            $networkStatus = 'CONGESTED';
        } elseif ($utilization > 50) {
            $networkStatus = 'BUSY';
        }
        
        return [
            'safe_gas_price' => (int)$data['safe_gas_price'],
            'propose_gas_price' => (int)$data['propose_gas_price'],
            'fast_gas_price' => (int)$data['fast_gas_price'],
            'base_fee' => $data['base_fee'],
            'network_utilization' => (float)$data['network_utilization'],
            'network_status' => $networkStatus,
            'timestamp' => $data['timestamp']
        ];
        
    } catch (Exception $e) {
        return null;
    }
}

function getWhaleActivity($pdo) {
    try {
        // Get whale movements in last hour
        $stmt = $pdo->query("
            SELECT 
                COUNT(*) as count,
                SUM(value_usd) as total_volume
            FROM token_transfers
            WHERE is_whale_movement = 1
                AND timestamp >= DATE_SUB(NOW(), INTERVAL 1 HOUR)
        ");
        
        $hourlyData = $stmt->fetch();
        
        // Get recent movements
        $stmt = $pdo->query("
            SELECT 
                token_symbol,
                value_usd,
                from_address,
                to_address,
                timestamp
            FROM token_transfers
            WHERE is_whale_movement = 1
            ORDER BY timestamp DESC
            LIMIT 5
        ");
        
        $recentMovements = $stmt->fetchAll();
        
        return [
            'last_hour_count' => (int)$hourlyData['count'],
            'total_volume_usd' => (float)$hourlyData['total_volume'],
            'recent_movements' => array_map(function($m) {
                return [
                    'token' => $m['token_symbol'],
                    'value_usd' => (float)$m['value_usd'],
                    'from' => Utils::truncateAddress($m['from_address']),
                    'to' => Utils::truncateAddress($m['to_address']),
                    'time_ago' => Utils::timeAgo($m['timestamp'])
                ];
            }, $recentMovements)
        ];
        
    } catch (Exception $e) {
        return null;
    }
}

function getDexVolume($pdo) {
    try {
        $stmt = $pdo->query("
            SELECT 
                dex_name,
                COUNT(*) as swap_count,
                SUM(amount_in_usd) as total_volume
            FROM dex_swaps
            WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 1 HOUR)
                AND amount_in_usd IS NOT NULL
            GROUP BY dex_name
            ORDER BY total_volume DESC
        ");
        
        $dexData = $stmt->fetchAll();
        
        return array_map(function($d) {
            return [
                'dex' => $d['dex_name'],
                'swap_count' => (int)$d['swap_count'],
                'total_volume_usd' => (float)$d['total_volume']
            ];
        }, $dexData);
        
    } catch (Exception $e) {
        return null;
    }
}
