<?php
/**
 * File: api/get_opportunities.php
 * Purpose: REST API endpoint to fetch MEV opportunities
 * Access: https://core.poipoe.com/mevdata/api/get_opportunities.php
 * Author: MEV Pipeline System
 * Last Modified: 2025-11-15
 */

// Error handling
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// CORS headers for frontend access
header('Access-Control-Allow-Origin: https://app.poipoe.com');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json');

// Handle preflight request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../helpers/Logger.php';
require_once __DIR__ . '/../helpers/Utils.php';

try {
    // Validate request method
    if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
        throw new Exception("Only GET requests allowed");
    }
    
    // Get and validate query parameters
    $type = $_GET['type'] ?? null;
    $minProfit = isset($_GET['min_profit']) ? (float)$_GET['min_profit'] : 0;
    $status = $_GET['status'] ?? 'pending';
    $limit = isset($_GET['limit']) ? min((int)$_GET['limit'], 100) : 50;
    $urgency = $_GET['urgency'] ?? null;
    
    // Build query
    $pdo = getDatabaseConnection();
    
    $sql = "
        SELECT 
            id,
            opportunity_type,
            status,
            token_addresses,
            dex_addresses,
            expected_profit_usd,
            gas_cost_usd,
            net_profit_usd,
            profit_percentage,
            confidence_score,
            urgency_level,
            related_tx_hashes,
            strategy_data,
            execution_steps,
            detected_at,
            expires_at,
            TIMESTAMPDIFF(SECOND, NOW(), expires_at) as time_remaining_seconds
        FROM mev_opportunities
        WHERE 1=1
    ";
    
    $params = [];
    
    // Filter by type
    if ($type) {
        $sql .= " AND opportunity_type = ?";
        $params[] = strtoupper($type);
    }
    
    // Filter by status
    if ($status) {
        $sql .= " AND status = ?";
        $params[] = $status;
    }
    
    // Filter by minimum profit
    if ($minProfit > 0) {
        $sql .= " AND net_profit_usd >= ?";
        $params[] = $minProfit;
    }
    
    // Filter by urgency
    if ($urgency) {
        $sql .= " AND urgency_level = ?";
        $params[] = strtoupper($urgency);
    }
    
    // Only show non-expired opportunities for pending status
    if ($status === 'pending') {
        $sql .= " AND expires_at > NOW()";
    }
    
    // Order by urgency and profit
    $sql .= " ORDER BY 
        FIELD(urgency_level, 'HIGH', 'MEDIUM', 'LOW'),
        net_profit_usd DESC,
        detected_at DESC
    ";
    
    $sql .= " LIMIT ?";
    $params[] = $limit;
    
    // Execute query
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $opportunities = $stmt->fetchAll();
    
    // Format opportunities for frontend
    $formattedOpportunities = [];
    foreach ($opportunities as $opp) {
        $strategyData = json_decode($opp['strategy_data'], true);
        $executionSteps = json_decode($opp['execution_steps'], true);
        
        $formatted = [
            'id' => (int)$opp['id'],
            'opportunity_type' => $opp['opportunity_type'],
            'status' => $opp['status'],
            'expected_profit_usd' => (float)$opp['expected_profit_usd'],
            'gas_cost_usd' => (float)$opp['gas_cost_usd'],
            'net_profit_usd' => (float)$opp['net_profit_usd'],
            'profit_percentage' => (float)$opp['profit_percentage'],
            'confidence_score' => (float)$opp['confidence_score'],
            'urgency_level' => $opp['urgency_level'],
            'time_remaining_seconds' => max(0, (int)$opp['time_remaining_seconds']),
            'detected_at' => $opp['detected_at'],
            'expires_at' => $opp['expires_at']
        ];
        
        // Add type-specific data
        if ($strategyData) {
            if ($opp['opportunity_type'] === 'ARBITRAGE') {
                $formatted['token_pair'] = $strategyData['token_pair'] ?? null;
                $formatted['dex_a'] = $strategyData['dex_a'] ?? null;
                $formatted['dex_b'] = $strategyData['dex_b'] ?? null;
                $formatted['price_diff'] = $strategyData['price_diff'] ?? null;
            } elseif ($opp['opportunity_type'] === 'SANDWICH') {
                $formatted['target_tx'] = $strategyData['target_tx'] ?? null;
                $formatted['dex'] = $strategyData['dex'] ?? null;
                $formatted['price_impact'] = $strategyData['price_impact'] ?? null;
            }
        }
        
        if ($executionSteps) {
            $formatted['execution_steps'] = $executionSteps;
        }
        
        $formattedOpportunities[] = $formatted;
    }
    
    // Get additional metadata
    $metadata = getMetadata($pdo);
    
    // Build response
    $response = [
        'success' => true,
        'count' => count($formattedOpportunities),
        'data' => $formattedOpportunities,
        'meta' => $metadata
    ];
    
    echo json_encode($response, JSON_PRETTY_PRINT);
    
} catch (Exception $e) {
    Logger::error("API error in get_opportunities", [
        'error' => $e->getMessage(),
        'request' => $_GET
    ]);
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'An error occurred while fetching opportunities',
        'message' => $e->getMessage()
    ]);
}

/**
 * Get metadata for response
 * @param PDO $pdo Database connection
 * @return array Metadata
 */
function getMetadata($pdo) {
    $metadata = [
        'current_gas_price' => null,
        'network_status' => 'UNKNOWN',
        'total_available' => 0
    ];
    
    try {
        // Get current gas price
        $stmt = $pdo->query("
            SELECT fast_gas_price, network_utilization
            FROM gas_tracker
            ORDER BY timestamp DESC
            LIMIT 1
        ");
        $gasData = $stmt->fetch();
        
        if ($gasData) {
            $metadata['current_gas_price'] = $gasData['fast_gas_price'] . ' gwei';
            
            // Determine network status
            $utilization = $gasData['network_utilization'];
            if ($utilization > 80) {
                $metadata['network_status'] = 'CONGESTED';
            } elseif ($utilization > 50) {
                $metadata['network_status'] = 'BUSY';
            } else {
                $metadata['network_status'] = 'NORMAL';
            }
        }
        
        // Get total available opportunities
        $stmt = $pdo->query("
            SELECT COUNT(*) as total
            FROM mev_opportunities
            WHERE status = 'pending' AND expires_at > NOW()
        ");
        $result = $stmt->fetch();
        $metadata['total_available'] = (int)$result['total'];
        
    } catch (Exception $e) {
        Logger::error("Failed to get metadata", ['error' => $e->getMessage()]);
    }
    
    return $metadata;
}
