<?php
/**
 * File: collectors/collect_dex_swaps.php
 * Purpose: Collect DEX swap events from major DEXes
 * Called by: Cron every minute
 * Author: MEV Pipeline System
 * Last Modified: 2025-11-15
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/etherscan.php';
require_once __DIR__ . '/../helpers/ApiClient.php';
require_once __DIR__ . '/../helpers/Logger.php';
require_once __DIR__ . '/../helpers/Utils.php';

/**
 * Collect DEX swap events
 * @return bool Success status
 */
function collectDexSwaps() {
    try {
        Logger::info("Starting DEX swaps collection");
        
        $pdo = getDatabaseConnection();
        $apiClient = new EtherscanApiClient();
        
        // Get latest block number we've processed
        $stmt = $pdo->query("SELECT MAX(block_number) as last_block FROM dex_swaps");
        $result = $stmt->fetch();
        $lastBlock = $result['last_block'] ?? 0;
        
        // Get current block number
        $currentBlock = $apiClient->getLatestBlockNumber();
        
        // Limit block range to avoid API limits
        $fromBlock = $lastBlock > 0 ? ($lastBlock + 1) : ($currentBlock - DEFAULT_BLOCK_RANGE);
        $toBlock = min($fromBlock + MAX_BLOCK_RANGE, $currentBlock);
        
        if ($fromBlock > $toBlock) {
            Logger::info("No new blocks to process");
            return true;
        }
        
        Logger::info("Processing DEX swaps", [
            'from_block' => $fromBlock,
            'to_block' => $toBlock,
            'block_range' => $toBlock - $fromBlock
        ]);
        
        $totalSwaps = 0;
        $largeSwaps = 0;
        
        // Process each major DEX
        foreach (MAJOR_DEX_CONTRACTS as $dexName => $contracts) {
            try {
                $swaps = collectDexSwapsForRouter(
                    $apiClient,
                    $dexName,
                    $contracts['router'],
                    $contracts['swap_signature'],
                    $fromBlock,
                    $toBlock
                );
                
                $totalSwaps += $swaps['total'];
                $largeSwaps += $swaps['large'];
                
                // Small delay between DEX queries
                usleep(100000); // 0.1 second
                
            } catch (Exception $e) {
                Logger::error("Failed to collect swaps for DEX", [
                    'dex' => $dexName,
                    'error' => $e->getMessage()
                ]);
                continue;
            }
        }
        
        Logger::info("DEX swaps collection completed", [
            'total_swaps' => $totalSwaps,
            'large_swaps' => $largeSwaps,
            'blocks_processed' => $toBlock - $fromBlock
        ]);
        
        return true;
        
    } catch (Exception $e) {
        Logger::error("DEX swaps collection failed", [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
        return false;
    }
}

/**
 * Collect swaps for a specific DEX router
 * @param EtherscanApiClient $apiClient API client
 * @param string $dexName DEX name
 * @param string $routerAddress Router contract address
 * @param string $swapSignature Swap event signature
 * @param int $fromBlock Starting block
 * @param int $toBlock Ending block
 * @return array Statistics
 */
function collectDexSwapsForRouter($apiClient, $dexName, $routerAddress, $swapSignature, $fromBlock, $toBlock) {
    $pdo = getDatabaseConnection();
    $totalSwaps = 0;
    $largeSwaps = 0;
    
    try {
        // Get swap events from this DEX
        $response = $apiClient->getLogs(
            $routerAddress,
            $fromBlock,
            $toBlock,
            $swapSignature
        );
        
        if (!isset($response['result']) || empty($response['result'])) {
            Logger::debug("No swap events found", [
                'dex' => $dexName,
                'from_block' => $fromBlock,
                'to_block' => $toBlock
            ]);
            return ['total' => 0, 'large' => 0];
        }
        
        $events = $response['result'];
        
        foreach ($events as $event) {
            $processed = processSwapEvent($event, $dexName, $routerAddress);
            if ($processed) {
                $totalSwaps++;
                if ($processed['is_large']) {
                    $largeSwaps++;
                }
            }
        }
        
        Logger::debug("Swaps collected for DEX", [
            'dex' => $dexName,
            'total' => $totalSwaps,
            'large' => $largeSwaps
        ]);
        
    } catch (Exception $e) {
        Logger::error("Failed to collect swaps for router", [
            'dex' => $dexName,
            'router' => $routerAddress,
            'error' => $e->getMessage()
        ]);
    }
    
    return ['total' => $totalSwaps, 'large' => $largeSwaps];
}

/**
 * Process individual swap event
 * @param array $event Event log data
 * @param string $dexName DEX name
 * @param string $dexAddress DEX contract address
 * @return array|null Processed swap or null
 */
function processSwapEvent($event, $dexName, $dexAddress) {
    try {
        $pdo = getDatabaseConnection();
        
        // Check if swap already exists
        $checkStmt = $pdo->prepare("SELECT id FROM dex_swaps WHERE tx_hash = ?");
        $checkStmt->execute([$event['transactionHash']]);
        
        if ($checkStmt->fetch()) {
            return null; // Already processed
        }
        
        // Parse event data (simplified - in production, decode properly)
        // For now, store raw event in contract_events and basic swap info
        
        $blockNumber = hexdec($event['blockNumber']);
        $timestamp = time(); // Approximate - should get from block
        
        // Store raw event first
        storeContractEvent($event, $dexName);
        
        // Try to extract token addresses from topics
        $topics = $event['topics'] ?? [];
        $tokenIn = count($topics) > 1 ? $topics[1] : null;
        $tokenOut = count($topics) > 2 ? $topics[2] : null;
        
        // Simplified swap storage (in production, decode amounts properly)
        $stmt = $pdo->prepare("
            INSERT INTO dex_swaps 
            (tx_hash, block_number, timestamp, dex_name, dex_address, 
             token_in, token_out, trader_address, is_large_swap, created_at)
            VALUES (?, ?, FROM_UNIXTIME(?), ?, ?, ?, ?, ?, 0, NOW())
            ON DUPLICATE KEY UPDATE tx_hash = tx_hash
        ");
        
        $stmt->execute([
            $event['transactionHash'],
            $blockNumber,
            $timestamp,
            $dexName,
            $dexAddress,
            $tokenIn,
            $tokenOut,
            $event['address'] ?? null
        ]);
        
        return [
            'tx_hash' => $event['transactionHash'],
            'is_large' => false // Update when amounts are decoded
        ];
        
    } catch (Exception $e) {
        Logger::error("Failed to process swap event", [
            'tx_hash' => $event['transactionHash'] ?? 'unknown',
            'error' => $e->getMessage()
        ]);
        return null;
    }
}

/**
 * Store raw contract event
 * @param array $event Event data
 * @param string $eventName Event name
 */
function storeContractEvent($event, $eventName) {
    try {
        $pdo = getDatabaseConnection();
        
        $stmt = $pdo->prepare("
            INSERT INTO contract_events 
            (tx_hash, block_number, log_index, timestamp, contract_address, 
             event_name, event_signature, topics, data, created_at)
            VALUES (?, ?, ?, NOW(), ?, ?, ?, ?, ?, NOW())
        ");
        
        $blockNumber = hexdec($event['blockNumber'] ?? '0');
        $logIndex = hexdec($event['logIndex'] ?? '0');
        
        $stmt->execute([
            $event['transactionHash'] ?? null,
            $blockNumber,
            $logIndex,
            $event['address'] ?? null,
            $eventName,
            $event['topics'][0] ?? null,
            json_encode($event['topics'] ?? []),
            $event['data'] ?? null
        ]);
        
    } catch (Exception $e) {
        // Silently fail - event storage is secondary
        Logger::debug("Failed to store contract event", [
            'error' => $e->getMessage()
        ]);
    }
}

/**
 * Get recent large swaps
 * @param int $limit Number of swaps to retrieve
 * @return array Large swaps
 */
function getRecentLargeSwaps($limit = 10) {
    try {
        $pdo = getDatabaseConnection();
        
        $stmt = $pdo->prepare("
            SELECT 
                tx_hash,
                dex_name,
                amount_in_usd,
                amount_out_usd,
                timestamp
            FROM dex_swaps
            WHERE is_large_swap = 1
            ORDER BY timestamp DESC
            LIMIT ?
        ");
        $stmt->execute([$limit]);
        
        return $stmt->fetchAll();
        
    } catch (Exception $e) {
        Logger::error("Failed to get large swaps", [
            'error' => $e->getMessage()
        ]);
        return [];
    }
}

// CLI execution
if (php_sapi_name() === 'cli') {
    $testMode = false;
    
    if (isset($argv) && in_array('--test', $argv)) {
        $testMode = true;
        echo "========================================\n";
        echo "DEX Swaps Collector - Test Mode\n";
        echo "========================================\n\n";
    }
    
    $success = collectDexSwaps();
    
    if ($testMode) {
        if ($success) {
            echo "✓ DEX swaps collected successfully\n\n";
            
            // Display recent swaps
            try {
                $pdo = getDatabaseConnection();
                $stmt = $pdo->query("
                    SELECT COUNT(*) as total,
                           COUNT(CASE WHEN is_large_swap = 1 THEN 1 END) as large_swaps
                    FROM dex_swaps
                ");
                $stats = $stmt->fetch();
                
                echo "DEX Swaps Statistics:\n";
                echo "  - Total swaps: " . $stats['total'] . "\n";
                echo "  - Large swaps: " . $stats['large_swaps'] . "\n";
            } catch (Exception $e) {
                echo "Error getting statistics\n";
            }
        } else {
            echo "✗ DEX swaps collection failed\n";
            echo "Check logs for details\n";
        }
        
        echo "\nTest complete.\n";
    }
    
    exit($success ? 0 : 1);
}
