<?php
/**
 * File: collectors/collect_gas_tracker.php
 * Purpose: Collect real-time gas price data from Etherscan Gas Oracle
 * Called by: Cron every minute
 * Author: MEV Pipeline System
 * Last Modified: 2025-11-15
 */

// Error reporting configuration
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

// Include dependencies
require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/etherscan.php';
require_once __DIR__ . '/../helpers/ApiClient.php';
require_once __DIR__ . '/../helpers/Logger.php';
require_once __DIR__ . '/../helpers/Utils.php';

/**
 * Collect and store gas price data
 * @return bool Success status
 */
function collectGasData() {
    try {
        Logger::info("Starting gas data collection");
        
        // Step 1: Initialize API client
        $apiClient = new EtherscanApiClient();
        
        // Step 2: Fetch gas oracle data
        $response = $apiClient->getGasOracle();
        
        // Step 3: Validate response
        if (!isset($response['status']) || $response['status'] != '1') {
            throw new Exception("Invalid API response: " . json_encode($response));
        }
        
        if (!isset($response['result'])) {
            throw new Exception("Missing result in API response");
        }
        
        $gasData = $response['result'];
        
        // Step 4: Extract and validate gas prices
        $safeGasPrice = isset($gasData['SafeGasPrice']) ? (int)$gasData['SafeGasPrice'] : null;
        $proposeGasPrice = isset($gasData['ProposeGasPrice']) ? (int)$gasData['ProposeGasPrice'] : null;
        $fastGasPrice = isset($gasData['FastGasPrice']) ? (int)$gasData['FastGasPrice'] : null;
        $baseFee = $gasData['suggestBaseFee'] ?? null;
        $gasUsedRatio = $gasData['gasUsedRatio'] ?? null;
        
        // Calculate network utilization from gas used ratio
        $networkUtilization = null;
        if ($gasUsedRatio) {
            $ratios = explode(',', $gasUsedRatio);
            if (!empty($ratios)) {
                $avgRatio = array_sum(array_map('floatval', $ratios)) / count($ratios);
                $networkUtilization = round($avgRatio * 100, 2);
            }
        }
        
        // Step 5: Insert into database
        $pdo = getDatabaseConnection();
        
        // Check if we already have data for this minute (avoid duplicates)
        $currentMinute = date('Y-m-d H:i:00');
        $checkStmt = $pdo->prepare("SELECT id FROM gas_tracker WHERE timestamp = ?");
        $checkStmt->execute([$currentMinute]);
        
        if ($checkStmt->fetch()) {
            Logger::info("Gas data already exists for this minute, updating", [
                'timestamp' => $currentMinute
            ]);
            
            // Update existing record
            $stmt = $pdo->prepare("
                UPDATE gas_tracker 
                SET safe_gas_price = ?,
                    propose_gas_price = ?,
                    fast_gas_price = ?,
                    base_fee = ?,
                    gas_used_ratio = ?,
                    network_utilization = ?
                WHERE timestamp = ?
            ");
            
            $stmt->execute([
                $safeGasPrice,
                $proposeGasPrice,
                $fastGasPrice,
                $baseFee,
                $gasUsedRatio,
                $networkUtilization,
                $currentMinute
            ]);
        } else {
            // Insert new record
            $stmt = $pdo->prepare("
                INSERT INTO gas_tracker 
                (timestamp, safe_gas_price, propose_gas_price, fast_gas_price, base_fee, gas_used_ratio, network_utilization) 
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $currentMinute,
                $safeGasPrice,
                $proposeGasPrice,
                $fastGasPrice,
                $baseFee,
                $gasUsedRatio,
                $networkUtilization
            ]);
        }
        
        // Step 6: Log success
        Logger::info("Gas data collected successfully", [
            'safe_gas' => $safeGasPrice,
            'propose_gas' => $proposeGasPrice,
            'fast_gas' => $fastGasPrice,
            'base_fee' => $baseFee,
            'network_utilization' => $networkUtilization
        ]);
        
        return true;
        
    } catch (ApiException $e) {
        Logger::error("Gas collection API error", [
            'error' => $e->getMessage(),
            'code' => $e->getCode(),
            'file' => __FILE__,
            'line' => __LINE__
        ]);
        return false;
        
    } catch (Exception $e) {
        Logger::error("Gas collection failed", [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString(),
            'file' => __FILE__,
            'line' => __LINE__
        ]);
        return false;
    }
}

/**
 * Clean old gas data (keep only last 7 days)
 * @return int Number of records deleted
 */
function cleanOldGasData() {
    try {
        $pdo = getDatabaseConnection();
        $days = GAS_DATA_RETENTION_DAYS;
        
        $stmt = $pdo->prepare("
            DELETE FROM gas_tracker 
            WHERE timestamp < DATE_SUB(NOW(), INTERVAL ? DAY)
        ");
        $stmt->execute([$days]);
        
        $deleted = $stmt->rowCount();
        
        if ($deleted > 0) {
            Logger::info("Cleaned old gas data", [
                'records_deleted' => $deleted,
                'retention_days' => $days
            ]);
        }
        
        return $deleted;
        
    } catch (Exception $e) {
        Logger::error("Failed to clean old gas data", [
            'error' => $e->getMessage()
        ]);
        return 0;
    }
}

/**
 * Get current gas price statistics
 * @return array Gas price stats
 */
function getGasStats() {
    try {
        $pdo = getDatabaseConnection();
        
        $stmt = $pdo->query("
            SELECT 
                AVG(safe_gas_price) as avg_safe,
                AVG(propose_gas_price) as avg_propose,
                AVG(fast_gas_price) as avg_fast,
                MAX(fast_gas_price) as max_fast,
                MIN(safe_gas_price) as min_safe,
                AVG(network_utilization) as avg_utilization
            FROM gas_tracker
            WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 1 HOUR)
        ");
        
        return $stmt->fetch();
        
    } catch (Exception $e) {
        Logger::error("Failed to get gas stats", [
            'error' => $e->getMessage()
        ]);
        return null;
    }
}

// Allow command-line execution and testing
if (php_sapi_name() === 'cli') {
    $testMode = false;
    
    // Check for test mode flag
    if (isset($argv) && in_array('--test', $argv)) {
        $testMode = true;
        echo "========================================\n";
        echo "Gas Tracker - Test Mode\n";
        echo "========================================\n\n";
    }
    
    // Run collection
    $success = collectGasData();
    
    if ($testMode) {
        if ($success) {
            echo "✓ Gas data collected successfully\n\n";
            
            // Display current gas prices
            $stats = getGasStats();
            if ($stats) {
                echo "Current Gas Price Statistics (Last Hour):\n";
                echo "  - Average Safe Gas: " . round($stats['avg_safe']) . " gwei\n";
                echo "  - Average Propose Gas: " . round($stats['avg_propose']) . " gwei\n";
                echo "  - Average Fast Gas: " . round($stats['avg_fast']) . " gwei\n";
                echo "  - Max Fast Gas: " . round($stats['max_fast']) . " gwei\n";
                echo "  - Min Safe Gas: " . round($stats['min_safe']) . " gwei\n";
                echo "  - Avg Network Utilization: " . round($stats['avg_utilization'], 2) . "%\n";
            }
        } else {
            echo "✗ Gas data collection failed\n";
            echo "Check logs for details\n";
        }
        
        echo "\nTest complete.\n";
    }
    
    exit($success ? 0 : 1);
}
