<?php
/**
 * File: collectors/collect_token_transfers.php
 * Purpose: Monitor token transfers from watched addresses
 * Called by: Cron every 2 minutes
 * Author: MEV Pipeline System
 * Last Modified: 2025-11-15
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/etherscan.php';
require_once __DIR__ . '/../helpers/ApiClient.php';
require_once __DIR__ . '/../helpers/Logger.php';
require_once __DIR__ . '/../helpers/Utils.php';

/**
 * Collect token transfers from watched addresses
 * @return bool Success status
 */
function collectTokenTransfers() {
    try {
        Logger::info("Starting token transfers collection");
        
        $pdo = getDatabaseConnection();
        $apiClient = new EtherscanApiClient();
        
        // Get active watched addresses
        $stmt = $pdo->query("
            SELECT address, name_tag, min_value_usd 
            FROM watched_addresses 
            WHERE is_active = 1 
            ORDER BY address_type, name_tag
        ");
        
        $watchedAddresses = $stmt->fetchAll();
        
        if (empty($watchedAddresses)) {
            Logger::warning("No watched addresses found");
            return true;
        }
        
        Logger::info("Processing watched addresses", [
            'count' => count($watchedAddresses)
        ]);
        
        $totalTransfers = 0;
        $whaleMovements = 0;
        
        foreach ($watchedAddresses as $watched) {
            try {
                // Get latest block we've processed for this address
                $lastBlockStmt = $pdo->prepare("
                    SELECT MAX(block_number) as last_block 
                    FROM token_transfers 
                    WHERE from_address = ? OR to_address = ?
                ");
                $lastBlockStmt->execute([$watched['address'], $watched['address']]);
                $lastBlock = $lastBlockStmt->fetch();
                $startBlock = $lastBlock['last_block'] ? ($lastBlock['last_block'] + 1) : 0;
                
                // Fetch token transfers for this address
                $response = $apiClient->getTokenTransfers(
                    $watched['address'],
                    $startBlock,
                    null,  // endBlock (null = latest)
                    null,  // contractAddress (null = all tokens)
                    1,     // page
                    100    // offset (max recent transfers)
                );
                
                if (!isset($response['result']) || empty($response['result'])) {
                    Logger::debug("No new transfers for address", [
                        'address' => Utils::truncateAddress($watched['address']),
                        'name' => $watched['name_tag']
                    ]);
                    continue;
                }
                
                $transfers = $response['result'];
                
                foreach ($transfers as $transfer) {
                    // Process each transfer
                    $processed = processTransfer($transfer, $watched['min_value_usd']);
                    if ($processed) {
                        $totalTransfers++;
                        if ($processed['is_whale']) {
                            $whaleMovements++;
                        }
                    }
                }
                
                // Update last activity timestamp
                $updateStmt = $pdo->prepare("
                    UPDATE watched_addresses 
                    SET last_activity = NOW() 
                    WHERE address = ?
                ");
                $updateStmt->execute([$watched['address']]);
                
                // Small delay to respect rate limits
                usleep(100000); // 0.1 second
                
            } catch (Exception $e) {
                Logger::error("Failed to process watched address", [
                    'address' => $watched['address'],
                    'name' => $watched['name_tag'],
                    'error' => $e->getMessage()
                ]);
                continue;
            }
        }
        
        Logger::info("Token transfers collection completed", [
            'total_transfers' => $totalTransfers,
            'whale_movements' => $whaleMovements,
            'addresses_processed' => count($watchedAddresses)
        ]);
        
        return true;
        
    } catch (Exception $e) {
        Logger::error("Token transfers collection failed", [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
        return false;
    }
}

/**
 * Process individual token transfer
 * @param array $transfer Transfer data from API
 * @param float $minValueUsd Minimum value threshold
 * @return array|null Processed transfer or null
 */
function processTransfer($transfer, $minValueUsd) {
    try {
        $pdo = getDatabaseConnection();
        
        // Check if transfer already exists
        $checkStmt = $pdo->prepare("
            SELECT id FROM token_transfers WHERE tx_hash = ? AND token_address = ?
        ");
        $checkStmt->execute([$transfer['hash'], $transfer['contractAddress']]);
        
        if ($checkStmt->fetch()) {
            return null; // Already processed
        }
        
        // Get token decimals
        $decimals = Utils::getTokenDecimals($transfer['contractAddress']);
        
        // Format token amount
        $formattedValue = Utils::formatTokenAmount($transfer['value'], $decimals);
        
        // Calculate USD value (approximate)
        $valueUsd = Utils::getApproximateUsdValue($transfer['contractAddress'], $formattedValue);
        
        // Determine if whale movement
        $isWhaleMovement = ($valueUsd >= $minValueUsd) ? 1 : 0;
        
        // Insert transfer record
        $stmt = $pdo->prepare("
            INSERT INTO token_transfers 
            (tx_hash, block_number, timestamp, from_address, to_address, token_address, 
             token_name, token_symbol, value, value_usd, gas_used, gas_price, is_whale_movement, created_at)
            VALUES (?, ?, FROM_UNIXTIME(?), ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
        ");
        
        $stmt->execute([
            $transfer['hash'],
            $transfer['blockNumber'],
            $transfer['timeStamp'],
            $transfer['from'],
            $transfer['to'],
            $transfer['contractAddress'],
            $transfer['tokenName'] ?? null,
            $transfer['tokenSymbol'] ?? null,
            $transfer['value'],
            $valueUsd,
            $transfer['gasUsed'] ?? null,
            $transfer['gasPrice'] ?? null,
            $isWhaleMovement
        ]);
        
        // Log significant whale movements
        if ($isWhaleMovement) {
            Logger::info("Whale movement detected", [
                'tx_hash' => $transfer['hash'],
                'token' => $transfer['tokenSymbol'] ?? 'UNKNOWN',
                'amount' => number_format($formattedValue, 2),
                'value_usd' => number_format($valueUsd, 2),
                'from' => Utils::truncateAddress($transfer['from']),
                'to' => Utils::truncateAddress($transfer['to'])
            ]);
        }
        
        return [
            'tx_hash' => $transfer['hash'],
            'value_usd' => $valueUsd,
            'is_whale' => $isWhaleMovement
        ];
        
    } catch (Exception $e) {
        Logger::error("Failed to process transfer", [
            'tx_hash' => $transfer['hash'] ?? 'unknown',
            'error' => $e->getMessage()
        ]);
        return null;
    }
}

/**
 * Get recent whale movements
 * @param int $limit Number of movements to retrieve
 * @return array Whale movements
 */
function getRecentWhaleMovements($limit = 10) {
    try {
        $pdo = getDatabaseConnection();
        
        $stmt = $pdo->prepare("
            SELECT 
                tx_hash,
                token_symbol,
                value,
                value_usd,
                from_address,
                to_address,
                timestamp
            FROM token_transfers
            WHERE is_whale_movement = 1
            ORDER BY timestamp DESC
            LIMIT ?
        ");
        $stmt->execute([$limit]);
        
        return $stmt->fetchAll();
        
    } catch (Exception $e) {
        Logger::error("Failed to get whale movements", [
            'error' => $e->getMessage()
        ]);
        return [];
    }
}

// CLI execution
if (php_sapi_name() === 'cli') {
    $testMode = false;
    
    if (isset($argv) && in_array('--test', $argv)) {
        $testMode = true;
        echo "========================================\n";
        echo "Token Transfers Collector - Test Mode\n";
        echo "========================================\n\n";
    }
    
    $success = collectTokenTransfers();
    
    if ($testMode) {
        if ($success) {
            echo "✓ Token transfers collected successfully\n\n";
            
            // Display recent whale movements
            $whaleMovements = getRecentWhaleMovements(5);
            if (!empty($whaleMovements)) {
                echo "Recent Whale Movements:\n";
                foreach ($whaleMovements as $movement) {
                    echo "  - " . $movement['token_symbol'] . ": $" . number_format($movement['value_usd'], 2) . "\n";
                    echo "    TX: " . Utils::truncateAddress($movement['tx_hash']) . "\n";
                    echo "    Time: " . $movement['timestamp'] . "\n\n";
                }
            } else {
                echo "No whale movements in database yet\n";
            }
        } else {
            echo "✗ Token transfers collection failed\n";
            echo "Check logs for details\n";
        }
        
        echo "\nTest complete.\n";
    }
    
    exit($success ? 0 : 1);
}
