<?php
/**
 * File: collectors/collect_whale_movements.php
 * Purpose: Track balance changes and large transactions of watched addresses
 * Called by: Cron every 2 minutes
 * Author: MEV Pipeline System
 * Last Modified: 2025-11-15
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/etherscan.php';
require_once __DIR__ . '/../helpers/ApiClient.php';
require_once __DIR__ . '/../helpers/Logger.php';
require_once __DIR__ . '/../helpers/Utils.php';

/**
 * Collect whale movements and balance changes
 * @return bool Success status
 */
function collectWhaleMovements() {
    try {
        Logger::info("Starting whale movements collection");
        
        $pdo = getDatabaseConnection();
        $apiClient = new EtherscanApiClient();
        
        // Get whale addresses
        $stmt = $pdo->query("
            SELECT address, name_tag, min_value_usd 
            FROM watched_addresses 
            WHERE is_active = 1 AND address_type IN ('WHALE', 'EXCHANGE')
            ORDER BY name_tag
        ");
        
        $whales = $stmt->fetchAll();
        
        if (empty($whales)) {
            Logger::warning("No whale addresses to monitor");
            return true;
        }
        
        Logger::info("Monitoring whale addresses", ['count' => count($whales)]);
        
        $totalMovements = 0;
        
        foreach ($whales as $whale) {
            try {
                // Get ETH balance
                $balance = $apiClient->getBalance($whale['address']);
                $balanceEth = Utils::weiToEther($balance);
                
                // Get recent transactions
                $transactions = $apiClient->getTransactions(
                    $whale['address'],
                    null, // startBlock
                    null, // endBlock  
                    1,    // page
                    10    // offset - get last 10 transactions
                );
                
                if (isset($transactions['result']) && !empty($transactions['result'])) {
                    foreach ($transactions['result'] as $tx) {
                        $processed = processWhaleTransaction($tx, $whale);
                        if ($processed) {
                            $totalMovements++;
                        }
                    }
                }
                
                // Update whale stats
                updateWhaleStats($whale['address'], $balanceEth);
                
                usleep(100000); // 0.1 second delay
                
            } catch (Exception $e) {
                Logger::error("Failed to process whale address", [
                    'address' => $whale['address'],
                    'name' => $whale['name_tag'],
                    'error' => $e->getMessage()
                ]);
                continue;
            }
        }
        
        Logger::info("Whale movements collection completed", [
            'movements' => $totalMovements,
            'whales_monitored' => count($whales)
        ]);
        
        return true;
        
    } catch (Exception $e) {
        Logger::error("Whale movements collection failed", [
            'error' => $e->getMessage()
        ]);
        return false;
    }
}

/**
 * Process whale transaction
 * @param array $tx Transaction data
 * @param array $whale Whale address data
 * @return bool Success status
 */
function processWhaleTransaction($tx, $whale) {
    try {
        $pdo = getDatabaseConnection();
        
        // Check if already processed
        $checkStmt = $pdo->prepare("
            SELECT id FROM token_transfers 
            WHERE tx_hash = ? AND (from_address = ? OR to_address = ?)
        ");
        $checkStmt->execute([$tx['hash'], $whale['address'], $whale['address']]);
        
        if ($checkStmt->fetch()) {
            return false; // Already processed
        }
        
        // Calculate transaction value in USD
        $valueEth = Utils::weiToEther($tx['value']);
        $ethPrice = Utils::getEthPrice();
        $valueUsd = bcmul($valueEth, (string)$ethPrice, 2);
        
        // Only track if above minimum threshold
        if ($valueUsd < $whale['min_value_usd']) {
            return false;
        }
        
        // Log whale movement
        Logger::info("Large whale transaction detected", [
            'whale' => $whale['name_tag'],
            'tx_hash' => $tx['hash'],
            'value_eth' => number_format($valueEth, 4),
            'value_usd' => number_format($valueUsd, 2),
            'from' => Utils::truncateAddress($tx['from']),
            'to' => Utils::truncateAddress($tx['to'])
        ]);
        
        return true;
        
    } catch (Exception $e) {
        Logger::error("Failed to process whale transaction", [
            'tx_hash' => $tx['hash'] ?? 'unknown',
            'error' => $e->getMessage()
        ]);
        return false;
    }
}

/**
 * Update whale statistics
 * @param string $address Whale address
 * @param string $balanceEth Current balance in ETH
 */
function updateWhaleStats($address, $balanceEth) {
    try {
        $pdo = getDatabaseConnection();
        
        $stmt = $pdo->prepare("
            UPDATE watched_addresses 
            SET last_activity = NOW()
            WHERE address = ?
        ");
        $stmt->execute([$address]);
        
    } catch (Exception $e) {
        Logger::error("Failed to update whale stats", [
            'address' => $address,
            'error' => $e->getMessage()
        ]);
    }
}

// CLI execution
if (php_sapi_name() === 'cli') {
    $testMode = false;
    
    if (isset($argv) && in_array('--test', $argv)) {
        $testMode = true;
        echo "========================================\n";
        echo "Whale Movements Collector - Test Mode\n";
        echo "========================================\n\n";
    }
    
    $success = collectWhaleMovements();
    
    if ($testMode) {
        echo $success ? "✓ Whale movements collected successfully\n" : "✗ Collection failed\n";
        echo "\nTest complete.\n";
    }
    
    exit($success ? 0 : 1);
}
