<?php
/**
 * File: config/database.php
 * Purpose: Database configuration and connection management
 * Author: MEV Pipeline System
 * Last Modified: 2025-11-15
 */

// Database credentials
define('DB_HOST', 'localhost');
define('DB_NAME', 'poipoeco_mevdata');
define('DB_USER', 'poipoeco_mevdata_user');
define('DB_PASS', '32TbgulnE7YOd2G0e5');
define('DB_CHARSET', 'utf8mb4');

// Rate limiting configuration
define('RATE_LIMIT_CALLS_PER_SECOND', 4);  // Conservative limit (Etherscan allows 5)
define('RATE_LIMIT_MAX_DAILY_CALLS', 80000);  // Conservative daily limit (100k max)

// Cache settings
define('CACHE_GAS_PRICE_SECONDS', 30);
define('CACHE_TOKEN_INFO_SECONDS', 300);
define('CACHE_BLOCK_INFO_SECONDS', 60);

// MEV detection thresholds
define('MEV_MIN_PROFIT_PERCENTAGE', 0.5);  // Minimum 0.5% profit
define('MEV_CONFIDENCE_THRESHOLD', 0.6);   // Minimum 60% confidence
define('WHALE_THRESHOLD_USD', 100000);     // $100k+ = whale movement
define('LARGE_SWAP_THRESHOLD_USD', 50000); // $50k+ = large swap

// Opportunity settings
define('OPPORTUNITY_EXPIRY_MINUTES', 5);   // Opportunities expire after 5 minutes
define('MAX_GAS_PRICE_GWEI', 200);         // Don't execute if gas > 200 gwei

// Data retention
define('DATA_RETENTION_DAYS', 30);         // Keep data for 30 days
define('GAS_DATA_RETENTION_DAYS', 7);      // Keep gas data for 7 days
define('LOG_RETENTION_DAYS', 30);          // Keep logs for 30 days

// Supported tokens (address => symbol)
define('SUPPORTED_TOKENS', [
    '0xA0b86991c6218b36c1d19D4a2e9Eb0cE3606eB48' => 'USDC',
    '0xdAC17F958D2ee523a2206206994597C13D831ec7' => 'USDT',
    '0xC02aaA39b223FE8D0A0e5C4F27eAD9083C756Cc2' => 'WETH',
    '0x6B175474E89094C44Da98b954EedeAC495271d0F' => 'DAI',
    '0x2260FAC5E5542a773Aa44fBCfeDf7C193bc2C599' => 'WBTC'
]);

// Major DEX contracts
define('MAJOR_DEX_CONTRACTS', [
    'UNISWAP_V2' => [
        'router' => '0x7a250d5630B4cF539739dF2C5dAcb4c659F2488D',
        'factory' => '0x5C69bEe701ef814a2B6a3EDD4B1652CB9cc5aA6f',
        'swap_signature' => '0xd78ad95fa46c994b6551d0da85fc275fe613ce37657fb8d5e3d130840159d822'
    ],
    'UNISWAP_V3' => [
        'router' => '0xE592427A0AEce92De3Edee1F18E0157C05861564',
        'factory' => '0x1F98431c8aD98523631AE4a59f267346ea31F984',
        'swap_signature' => '0xc42079f94a6350d7e6235f29174924f928cc2ac818eb64fed8004e115fbcca67'
    ],
    'SUSHISWAP' => [
        'router' => '0xd9e1cE17f2641f24aE83637ab66a2cca9C378B9F',
        'factory' => '0xC0AEe478e3658e2610c5F7A4A2E1777cE9e4f2Ac',
        'swap_signature' => '0xd78ad95fa46c994b6551d0da85fc275fe613ce37657fb8d5e3d130840159d822'
    ]
]);

/**
 * Get database connection (singleton pattern)
 * @return PDO Database connection
 * @throws PDOException if connection fails
 */
function getDatabaseConnection() {
    static $pdo = null;
    
    if ($pdo === null) {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                PDO::ATTR_PERSISTENT => false,  // Disable persistent connections for cron jobs
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci"
            ];
            
            $pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
            
        } catch (PDOException $e) {
            error_log("Database connection failed: " . $e->getMessage());
            throw new Exception("Database connection failed. Please check configuration.");
        }
    }
    
    return $pdo;
}

/**
 * Test database connection
 * @return bool True if connection successful
 */
function testDatabaseConnection() {
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->query("SELECT 1");
        return true;
    } catch (Exception $e) {
        error_log("Database test failed: " . $e->getMessage());
        return false;
    }
}

/**
 * Get system configuration value
 * @param string $key Configuration key
 * @param mixed $default Default value if not found
 * @return mixed Configuration value
 */
function getConfigValue($key, $default = null) {
    static $cache = [];
    
    if (isset($cache[$key])) {
        return $cache[$key];
    }
    
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->prepare("SELECT config_value, config_type FROM system_config WHERE config_key = ? AND is_active = 1");
        $stmt->execute([$key]);
        $result = $stmt->fetch();
        
        if (!$result) {
            return $default;
        }
        
        // Type casting
        $value = $result['config_value'];
        switch ($result['config_type']) {
            case 'integer':
                $value = (int)$value;
                break;
            case 'decimal':
            case 'float':
                $value = (float)$value;
                break;
            case 'boolean':
                $value = (bool)$value;
                break;
        }
        
        $cache[$key] = $value;
        return $value;
        
    } catch (Exception $e) {
        error_log("Error getting config value for {$key}: " . $e->getMessage());
        return $default;
    }
}

/**
 * Set system configuration value
 * @param string $key Configuration key
 * @param mixed $value Configuration value
 * @return bool Success status
 */
function setConfigValue($key, $value) {
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->prepare("
            UPDATE system_config 
            SET config_value = ?, last_updated = NOW() 
            WHERE config_key = ?
        ");
        $stmt->execute([$value, $key]);
        return true;
    } catch (Exception $e) {
        error_log("Error setting config value for {$key}: " . $e->getMessage());
        return false;
    }
}
