<?php
/**
 * File: cron/cron_master.php
 * Purpose: Master cron orchestrator for all MEV data collection tasks
 * Usage: php cron_master.php --task=TASK_NAME
 * Author: MEV Pipeline System
 * Last Modified: 2025-11-15
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('max_execution_time', 300); // 5 minutes max

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../helpers/Logger.php';
require_once __DIR__ . '/../helpers/Utils.php';

// Parse command line arguments
$options = getopt('', ['task:', 'action:']);
$task = $options['task'] ?? null;
$action = $options['action'] ?? null;

// Handle health check action
if ($action === 'health') {
    performHealthCheck();
    exit(0);
}

// Validate task parameter
if (!$task) {
    Logger::error("Cron master called without task parameter");
    echo json_encode(['success' => false, 'error' => 'Task parameter required']);
    exit(1);
}

// Prevent concurrent execution of the same task
$lockFile = __DIR__ . "/../cache/cron_lock_{$task}.lock";
ensureCacheDirectory();

if (file_exists($lockFile)) {
    $lockTime = filemtime($lockFile);
    $timeSinceLock = time() - $lockTime;
    
    // If lock is older than 10 minutes, assume previous run crashed and remove it
    if ($timeSinceLock > 600) {
        unlink($lockFile);
        Logger::warning("Removed stale lock file for task", ['task' => $task, 'age_seconds' => $timeSinceLock]);
    } else {
        Logger::warning("Task already running", ['task' => $task]);
        exit(0);
    }
}

// Create lock file
touch($lockFile);

try {
    Logger::info("Cron task started", ['task' => $task]);
    
    $startTime = microtime(true);
    $success = false;
    
    // Route to appropriate task
    switch ($task) {
        case 'gas_tracker':
            require_once __DIR__ . '/../collectors/collect_gas_tracker.php';
            $success = collectGasData();
            break;
            
        case 'dex_events':
            require_once __DIR__ . '/../collectors/collect_dex_swaps.php';
            $success = collectDexSwaps();
            break;
            
        case 'whale_movements':
            require_once __DIR__ . '/../collectors/collect_whale_movements.php';
            $success = collectWhaleMovements();
            break;
            
        case 'token_transfers':
            require_once __DIR__ . '/../collectors/collect_token_transfers.php';
            $success = collectTokenTransfers();
            break;
            
        case 'detect_opportunities':
            require_once __DIR__ . '/../processors/detect_arbitrage.php';
            require_once __DIR__ . '/../processors/detect_sandwich.php';
            
            $arbitrageSuccess = detectArbitrage();
            $sandwichSuccess = detectSandwich();
            $success = $arbitrageSuccess && $sandwichSuccess;
            break;
            
        case 'cleanup':
            $success = performCleanup();
            break;
            
        default:
            Logger::error("Unknown task", ['task' => $task]);
            throw new Exception("Unknown task: {$task}");
    }
    
    $executionTime = round((microtime(true) - $startTime) * 1000); // milliseconds
    
    Logger::info("Cron task completed", [
        'task' => $task,
        'success' => $success,
        'execution_time_ms' => $executionTime
    ]);
    
    // Remove lock file
    unlink($lockFile);
    
    exit($success ? 0 : 1);
    
} catch (Exception $e) {
    Logger::error("Cron task failed", [
        'task' => $task,
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
    ]);
    
    // Remove lock file
    if (file_exists($lockFile)) {
        unlink($lockFile);
    }
    
    exit(1);
}

/**
 * Perform system health check
 */
function performHealthCheck() {
    header('Content-Type: application/json');
    
    $health = [
        'overall_health' => 'HEALTHY',
        'timestamp' => date('Y-m-d H:i:s'),
        'checks' => []
    ];
    
    try {
        // Check database connection
        $pdo = getDatabaseConnection();
        $stmt = $pdo->query("SELECT 1");
        $health['checks']['database_connection'] = 'OK';
        
        // Check API key configured
        $health['checks']['api_key_configured'] = defined('ETHERSCAN_API_KEY') && !empty(ETHERSCAN_API_KEY);
        
        // Check API usage today
        $stmt = $pdo->query("
            SELECT COUNT(*) as count 
            FROM api_call_log 
            WHERE DATE(called_at) = CURDATE()
        ");
        $result = $stmt->fetch();
        $health['api_usage_today'] = (int)$result['count'];
        $health['api_limit_daily'] = RATE_LIMIT_MAX_DAILY_CALLS;
        $health['api_usage_percentage'] = round(($health['api_usage_today'] / RATE_LIMIT_MAX_DAILY_CALLS) * 100, 2);
        
        // Check rate limit status
        $health['checks']['rate_limit_status'] = $health['api_usage_today'] < RATE_LIMIT_MAX_DAILY_CALLS ? 'OK' : 'LIMIT_REACHED';
        
        // Check last gas update
        $stmt = $pdo->query("
            SELECT TIMESTAMPDIFF(SECOND, MAX(timestamp), NOW()) as seconds_ago
            FROM gas_tracker
        ");
        $result = $stmt->fetch();
        $secondsAgo = (int)$result['seconds_ago'];
        $health['last_gas_update'] = $secondsAgo . ' seconds ago';
        $health['checks']['gas_tracker_active'] = $secondsAgo < 300 ? 'OK' : 'STALE'; // Should update every minute
        
        // Check last opportunity detected
        $stmt = $pdo->query("
            SELECT TIMESTAMPDIFF(SECOND, MAX(detected_at), NOW()) as seconds_ago
            FROM mev_opportunities
        ");
        $result = $stmt->fetch();
        $opportunityAge = $result['seconds_ago'];
        if ($opportunityAge !== null) {
            $health['last_opportunity_detected'] = Utils::timeAgo(date('Y-m-d H:i:s', time() - $opportunityAge));
        } else {
            $health['last_opportunity_detected'] = 'Never';
        }
        
        // Check errors in last hour
        $stmt = $pdo->query("
            SELECT COUNT(*) as count
            FROM error_log
            WHERE created_at >= DATE_SUB(NOW(), INTERVAL 1 HOUR)
                AND severity IN ('ERROR', 'CRITICAL')
        ");
        $result = $stmt->fetch();
        $health['errors_last_hour'] = (int)$result['count'];
        $health['checks']['error_rate'] = $health['errors_last_hour'] < 10 ? 'OK' : 'HIGH';
        
        // Determine overall health
        $unhealthyChecks = 0;
        foreach ($health['checks'] as $check => $status) {
            if ($status !== 'OK' && $status !== true) {
                $unhealthyChecks++;
            }
        }
        
        if ($unhealthyChecks === 0) {
            $health['overall_health'] = 'HEALTHY';
        } elseif ($unhealthyChecks < 3) {
            $health['overall_health'] = 'DEGRADED';
        } else {
            $health['overall_health'] = 'UNHEALTHY';
        }
        
    } catch (Exception $e) {
        $health['overall_health'] = 'UNHEALTHY';
        $health['error'] = $e->getMessage();
    }
    
    echo json_encode($health, JSON_PRETTY_PRINT);
}

/**
 * Perform cleanup tasks
 * @return bool Success status
 */
function performCleanup() {
    Logger::info("Starting cleanup tasks");
    
    $success = true;
    
    try {
        $pdo = getDatabaseConnection();
        
        // Clean old gas data (keep 7 days)
        $stmt = $pdo->prepare("
            DELETE FROM gas_tracker 
            WHERE timestamp < DATE_SUB(NOW(), INTERVAL ? DAY)
        ");
        $stmt->execute([GAS_DATA_RETENTION_DAYS]);
        $gasDeleted = $stmt->rowCount();
        
        // Clean expired opportunities (keep 30 days)
        $stmt = $pdo->prepare("
            DELETE FROM mev_opportunities 
            WHERE status = 'expired' 
                AND expires_at < DATE_SUB(NOW(), INTERVAL ? DAY)
        ");
        $stmt->execute([DATA_RETENTION_DAYS]);
        $oppDeleted = $stmt->rowCount();
        
        // Update expired opportunities
        $stmt = $pdo->query("
            UPDATE mev_opportunities 
            SET status = 'expired' 
            WHERE status = 'pending' 
                AND expires_at < NOW()
        ");
        $oppExpired = $stmt->rowCount();
        
        // Clean old API logs (keep 30 days)
        $stmt = $pdo->prepare("
            DELETE FROM api_call_log 
            WHERE called_at < DATE_SUB(NOW(), INTERVAL ? DAY)
        ");
        $stmt->execute([LOG_RETENTION_DAYS]);
        $apiLogsDeleted = $stmt->rowCount();
        
        // Clean old contract events (keep 14 days)
        $stmt = $pdo->query("
            DELETE FROM contract_events 
            WHERE created_at < DATE_SUB(NOW(), INTERVAL 14 DAY)
        ");
        $eventsDeleted = $stmt->rowCount();
        
        // Clean old log files
        Logger::cleanOldLogs(LOG_RETENTION_DAYS);
        
        Logger::info("Cleanup completed", [
            'gas_records_deleted' => $gasDeleted,
            'opportunities_deleted' => $oppDeleted,
            'opportunities_expired' => $oppExpired,
            'api_logs_deleted' => $apiLogsDeleted,
            'events_deleted' => $eventsDeleted
        ]);
        
    } catch (Exception $e) {
        Logger::error("Cleanup failed", [
            'error' => $e->getMessage()
        ]);
        $success = false;
    }
    
    return $success;
}

/**
 * Ensure cache directory exists
 */
function ensureCacheDirectory() {
    $cacheDir = __DIR__ . '/../cache';
    if (!is_dir($cacheDir)) {
        mkdir($cacheDir, 0755, true);
        
        // Create .htaccess to protect cache directory
        file_put_contents($cacheDir . '/.htaccess', "Deny from all\n");
    }
}
