<?php
/**
 * File: helpers/ApiClient.php
 * Purpose: Etherscan API client with rate limiting and retry logic
 * Author: MEV Pipeline System
 * Last Modified: 2025-11-15
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/etherscan.php';
require_once __DIR__ . '/RateLimiter.php';
require_once __DIR__ . '/Logger.php';

class EtherscanApiClient {
    
    private $apiKey;
    private $baseUrl;
    private $rateLimiter;
    private $pdo;
    private $timeout;
    private $retryAttempts;
    private $retryDelay;
    
    public function __construct() {
        $this->apiKey = ETHERSCAN_API_KEY;
        $this->baseUrl = ETHERSCAN_API_URL;
        $this->timeout = ETHERSCAN_TIMEOUT;
        $this->retryAttempts = ETHERSCAN_RETRY_ATTEMPTS;
        $this->retryDelay = ETHERSCAN_RETRY_DELAY;
        $this->rateLimiter = new RateLimiter();
        $this->pdo = getDatabaseConnection();
    }
    
    /**
     * Get current gas prices from Etherscan Gas Oracle
     * @return array Gas price data
     * @throws ApiException
     */
    public function getGasOracle() {
        return $this->makeRequest([
            'module' => 'gastracker',
            'action' => 'gasoracle'
        ]);
    }
    
    /**
     * Get token transfers for an address
     * @param string $address Ethereum address
     * @param int $startBlock Starting block (optional)
     * @param int $endBlock Ending block (optional)
     * @param string $contractAddress Token contract address (optional)
     * @param int $page Page number (optional)
     * @param int $offset Results per page (optional, max 10000)
     * @return array Token transfer records
     */
    public function getTokenTransfers($address, $startBlock = null, $endBlock = null, $contractAddress = null, $page = 1, $offset = 100) {
        $params = [
            'module' => 'account',
            'action' => 'tokentx',
            'address' => $address,
            'page' => $page,
            'offset' => $offset,
            'sort' => 'desc'
        ];
        
        if ($startBlock !== null) $params['startblock'] = $startBlock;
        if ($endBlock !== null) $params['endblock'] = $endBlock;
        if ($contractAddress !== null) $params['contractaddress'] = $contractAddress;
        
        return $this->makeRequest($params);
    }
    
    /**
     * Get event logs from contract
     * @param string $address Contract address
     * @param int $fromBlock Starting block
     * @param int $toBlock Ending block
     * @param string $topic0 Event signature hash (optional)
     * @return array Event log records
     */
    public function getLogs($address, $fromBlock, $toBlock, $topic0 = null) {
        $params = [
            'module' => 'logs',
            'action' => 'getLogs',
            'address' => $address,
            'fromBlock' => $fromBlock,
            'toBlock' => $toBlock
        ];
        
        if ($topic0 !== null) {
            $params['topic0'] = $topic0;
        }
        
        return $this->makeRequest($params);
    }
    
    /**
     * Get ETH balance for address
     * @param string $address Ethereum address
     * @return string Balance in wei
     */
    public function getBalance($address) {
        $result = $this->makeRequest([
            'module' => 'account',
            'action' => 'balance',
            'address' => $address,
            'tag' => 'latest'
        ]);
        
        return $result['result'] ?? '0';
    }
    
    /**
     * Get multiple ETH balances at once
     * @param array $addresses Array of Ethereum addresses (max 20)
     * @return array Balances
     */
    public function getBalanceMulti($addresses) {
        if (count($addresses) > 20) {
            throw new ApiException("Maximum 20 addresses allowed for balance multi query");
        }
        
        return $this->makeRequest([
            'module' => 'account',
            'action' => 'balancemulti',
            'address' => implode(',', $addresses),
            'tag' => 'latest'
        ]);
    }
    
    /**
     * Get transaction by hash
     * @param string $txhash Transaction hash
     * @return array Transaction details
     */
    public function getTransactionByHash($txhash) {
        return $this->makeRequest([
            'module' => 'proxy',
            'action' => 'eth_getTransactionByHash',
            'txhash' => $txhash
        ]);
    }
    
    /**
     * Get transaction receipt
     * @param string $txhash Transaction hash
     * @return array Transaction receipt
     */
    public function getTransactionReceipt($txhash) {
        return $this->makeRequest([
            'module' => 'proxy',
            'action' => 'eth_getTransactionReceipt',
            'txhash' => $txhash
        ]);
    }
    
    /**
     * Get latest block number
     * @return int Block number
     */
    public function getLatestBlockNumber() {
        $result = $this->makeRequest([
            'module' => 'proxy',
            'action' => 'eth_blockNumber'
        ]);
        
        return hexdec($result['result'] ?? '0');
    }
    
    /**
     * Get block by number
     * @param int $blockNumber Block number
     * @return array Block data
     */
    public function getBlockByNumber($blockNumber) {
        $blockHex = '0x' . dechex($blockNumber);
        
        return $this->makeRequest([
            'module' => 'proxy',
            'action' => 'eth_getBlockByNumber',
            'tag' => $blockHex,
            'boolean' => 'true'
        ]);
    }
    
    /**
     * Get normal transactions for an address
     * @param string $address Ethereum address
     * @param int $startBlock Starting block (optional)
     * @param int $endBlock Ending block (optional)
     * @param int $page Page number (optional)
     * @param int $offset Results per page (optional)
     * @return array Transaction records
     */
    public function getTransactions($address, $startBlock = null, $endBlock = null, $page = 1, $offset = 100) {
        $params = [
            'module' => 'account',
            'action' => 'txlist',
            'address' => $address,
            'page' => $page,
            'offset' => $offset,
            'sort' => 'desc'
        ];
        
        if ($startBlock !== null) $params['startblock'] = $startBlock;
        if ($endBlock !== null) $params['endblock'] = $endBlock;
        
        return $this->makeRequest($params);
    }
    
    /**
     * Make API request with rate limiting and retry logic
     * @param array $params Request parameters
     * @param int $retryCount Current retry attempt
     * @return array API response
     * @throws ApiException
     */
    private function makeRequest($params, $retryCount = 0) {
        $startTime = microtime(true);
        
        try {
            // Check rate limits
            if (!$this->rateLimiter->allowRequest()) {
                throw new ApiException("Rate limit exceeded", 429);
            }
            
            // Add API key
            $params['apikey'] = $this->apiKey;
            
            // Build URL
            $url = $this->baseUrl . '?' . http_build_query($params);
            
            // Make request
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => $this->timeout,
                CURLOPT_CONNECTTIMEOUT => 10,
                CURLOPT_SSL_VERIFYPEER => true,
                CURLOPT_SSL_VERIFYHOST => 2,
                CURLOPT_USERAGENT => 'MEV-Pipeline/1.0',
                CURLOPT_HTTPHEADER => [
                    'Accept: application/json'
                ]
            ]);
            
            $response = curl_exec($ch);
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $responseTime = round((microtime(true) - $startTime) * 1000);
            $curlError = curl_error($ch);
            $curlErrno = curl_errno($ch);
            
            curl_close($ch);
            
            // Handle cURL errors
            if ($curlErrno) {
                throw new ApiException("cURL error ({$curlErrno}): {$curlError}", $curlErrno);
            }
            
            // Handle HTTP errors
            if ($httpCode !== 200) {
                throw new ApiException("HTTP error: {$httpCode}", $httpCode);
            }
            
            // Parse response
            $data = json_decode($response, true);
            
            if ($data === null) {
                throw new ApiException("Invalid JSON response: " . substr($response, 0, 200));
            }
            
            // Log API call
            $this->logApiCall($params, $httpCode, $responseTime, null);
            
            // Check for API errors
            if (isset($data['status']) && $data['status'] == '0') {
                $message = $data['message'] ?? $data['result'] ?? 'Unknown API error';
                
                // Handle rate limit errors with retry
                if (stripos($message, 'rate limit') !== false || stripos($message, 'max rate') !== false) {
                    if ($retryCount < $this->retryAttempts) {
                        Logger::warning("Rate limit hit, retrying", [
                            'retry' => $retryCount + 1,
                            'max_retries' => $this->retryAttempts
                        ]);
                        sleep($this->retryDelay * ($retryCount + 1)); // Exponential backoff
                        return $this->makeRequest($params, $retryCount + 1);
                    }
                    throw new ApiException("API rate limit: {$message}", 429);
                }
                
                // Handle "No transactions found" as valid empty result
                if (stripos($message, 'No records found') !== false || 
                    stripos($message, 'No transactions found') !== false) {
                    return [
                        'status' => '1',
                        'message' => 'OK',
                        'result' => []
                    ];
                }
                
                throw new ApiException("API error: {$message}");
            }
            
            return $data;
            
        } catch (ApiException $e) {
            // Log error
            $this->logApiCall($params, $e->getCode(), $responseTime ?? 0, $e->getMessage());
            
            Logger::error("API request failed", [
                'module' => $params['module'] ?? '',
                'action' => $params['action'] ?? '',
                'error' => $e->getMessage(),
                'retry' => $retryCount,
                'code' => $e->getCode()
            ]);
            
            // Retry on network errors (but not rate limits)
            if ($retryCount < $this->retryAttempts && $e->getCode() !== 429) {
                Logger::info("Retrying API request", [
                    'retry' => $retryCount + 1,
                    'max_retries' => $this->retryAttempts
                ]);
                sleep($this->retryDelay);
                return $this->makeRequest($params, $retryCount + 1);
            }
            
            throw $e;
        }
    }
    
    /**
     * Log API call to database
     * @param array $params Request parameters
     * @param int $status HTTP status code
     * @param int $responseTime Response time in milliseconds
     * @param string $errorMessage Error message (if any)
     */
    private function logApiCall($params, $status, $responseTime, $errorMessage = null) {
        try {
            // Remove API key from logged parameters
            $logParams = $params;
            unset($logParams['apikey']);
            
            $stmt = $this->pdo->prepare("
                INSERT INTO api_call_log 
                (endpoint, module, action, parameters, response_status, response_time_ms, error_message, called_at) 
                VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
            ");
            
            $stmt->execute([
                $this->baseUrl,
                $params['module'] ?? '',
                $params['action'] ?? '',
                json_encode($logParams),
                $status,
                $responseTime,
                $errorMessage
            ]);
            
        } catch (Exception $e) {
            // Silently fail logging to avoid blocking API calls
            error_log("API call logging failed: " . $e->getMessage());
        }
    }
}

/**
 * Custom exception for API errors
 */
class ApiException extends Exception {
    public function __construct($message = "", $code = 0, Throwable $previous = null) {
        parent::__construct($message, $code, $previous);
    }
}
