<?php
/**
 * File: helpers/Logger.php
 * Purpose: Centralized logging system with file and database logging
 * Author: MEV Pipeline System
 * Last Modified: 2025-11-15
 */

require_once __DIR__ . '/../config/database.php';

class Logger {
    
    const DEBUG = 'DEBUG';
    const INFO = 'INFO';
    const WARNING = 'WARNING';
    const ERROR = 'ERROR';
    const CRITICAL = 'CRITICAL';
    
    private static $pdo;
    private static $logDir;
    private static $maxFileSize = 10485760; // 10MB
    
    /**
     * Initialize logger
     */
    public static function init() {
        if (self::$pdo === null) {
            self::$pdo = getDatabaseConnection();
            self::$logDir = __DIR__ . '/../logs/';
            
            // Create logs directory if it doesn't exist
            if (!is_dir(self::$logDir)) {
                mkdir(self::$logDir, 0755, true);
            }
            
            // Create .htaccess to protect logs directory
            $htaccess = self::$logDir . '.htaccess';
            if (!file_exists($htaccess)) {
                file_put_contents($htaccess, "Deny from all\n");
            }
        }
    }
    
    /**
     * Log debug message
     * @param string $message Log message
     * @param array $context Additional context data
     */
    public static function debug($message, $context = []) {
        self::log(self::DEBUG, $message, $context);
    }
    
    /**
     * Log info message
     * @param string $message Log message
     * @param array $context Additional context data
     */
    public static function info($message, $context = []) {
        self::log(self::INFO, $message, $context);
    }
    
    /**
     * Log warning
     * @param string $message Log message
     * @param array $context Additional context data
     */
    public static function warning($message, $context = []) {
        self::log(self::WARNING, $message, $context);
    }
    
    /**
     * Log error
     * @param string $message Log message
     * @param array $context Additional context data
     */
    public static function error($message, $context = []) {
        self::log(self::ERROR, $message, $context);
    }
    
    /**
     * Log critical error
     * @param string $message Log message
     * @param array $context Additional context data
     */
    public static function critical($message, $context = []) {
        self::log(self::CRITICAL, $message, $context);
    }
    
    /**
     * Main logging method
     * @param string $level Log level
     * @param string $message Log message
     * @param array $context Additional context data
     */
    private static function log($level, $message, $context = []) {
        self::init();
        
        $timestamp = date('Y-m-d H:i:s');
        $contextJson = !empty($context) ? json_encode($context, JSON_UNESCAPED_SLASHES) : '';
        
        // Log to file
        self::logToFile($level, $message, $contextJson, $timestamp);
        
        // Log to database (only WARNING, ERROR, CRITICAL)
        if (in_array($level, [self::WARNING, self::ERROR, self::CRITICAL])) {
            self::logToDatabase($level, $message, $contextJson);
        }
    }
    
    /**
     * Log to file
     * @param string $level Log level
     * @param string $message Log message
     * @param string $contextJson Context as JSON
     * @param string $timestamp Timestamp
     */
    private static function logToFile($level, $message, $contextJson, $timestamp) {
        try {
            $logFile = self::$logDir . 'app_' . date('Y-m-d') . '.log';
            
            // Check file size and rotate if necessary
            if (file_exists($logFile) && filesize($logFile) > self::$maxFileSize) {
                $rotatedFile = self::$logDir . 'app_' . date('Y-m-d') . '_' . time() . '.log';
                rename($logFile, $rotatedFile);
            }
            
            $logEntry = "[{$timestamp}] [{$level}] {$message}";
            
            if (!empty($contextJson)) {
                $logEntry .= " | Context: {$contextJson}";
            }
            
            $logEntry .= PHP_EOL;
            
            file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
            
        } catch (Exception $e) {
            error_log("Logger file write failed: " . $e->getMessage());
        }
    }
    
    /**
     * Log to database
     * @param string $level Log level
     * @param string $message Log message
     * @param string $contextJson Context as JSON
     */
    private static function logToDatabase($level, $message, $contextJson) {
        try {
            // Get caller information
            $backtrace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 4);
            $caller = $backtrace[3] ?? $backtrace[2] ?? [];
            
            $stmt = self::$pdo->prepare("
                INSERT INTO error_log 
                (error_type, error_message, context_data, severity, file_path, line_number, created_at) 
                VALUES (?, ?, ?, ?, ?, ?, NOW())
            ");
            
            $stmt->execute([
                'APPLICATION',
                substr($message, 0, 1000),  // Limit message length
                $contextJson,
                $level,
                $caller['file'] ?? '',
                $caller['line'] ?? 0
            ]);
            
        } catch (Exception $e) {
            // Silently fail database logging to avoid infinite loops
            error_log("Logger database write failed: " . $e->getMessage());
        }
    }
    
    /**
     * Clean old log files
     * @param int $days Number of days to keep
     * @return int Number of files deleted
     */
    public static function cleanOldLogs($days = LOG_RETENTION_DAYS) {
        self::init();
        
        $deleted = 0;
        $cutoffTime = time() - ($days * 86400);
        
        try {
            $files = glob(self::$logDir . '*.log');
            
            foreach ($files as $file) {
                if (filemtime($file) < $cutoffTime) {
                    if (unlink($file)) {
                        $deleted++;
                    }
                }
            }
            
            // Also clean database logs
            $stmt = self::$pdo->prepare("
                DELETE FROM error_log 
                WHERE created_at < DATE_SUB(NOW(), INTERVAL ? DAY)
            ");
            $stmt->execute([$days]);
            
            self::info("Cleaned old logs", [
                'files_deleted' => $deleted,
                'db_rows_deleted' => $stmt->rowCount(),
                'retention_days' => $days
            ]);
            
        } catch (Exception $e) {
            self::error("Failed to clean old logs", [
                'error' => $e->getMessage()
            ]);
        }
        
        return $deleted;
    }
    
    /**
     * Get recent errors from database
     * @param int $limit Number of errors to retrieve
     * @param string $severity Severity filter
     * @return array Recent errors
     */
    public static function getRecentErrors($limit = 50, $severity = null) {
        self::init();
        
        try {
            $sql = "SELECT * FROM error_log";
            $params = [];
            
            if ($severity) {
                $sql .= " WHERE severity = ?";
                $params[] = $severity;
            }
            
            $sql .= " ORDER BY created_at DESC LIMIT ?";
            $params[] = $limit;
            
            $stmt = self::$pdo->prepare($sql);
            $stmt->execute($params);
            
            return $stmt->fetchAll();
            
        } catch (Exception $e) {
            error_log("Failed to get recent errors: " . $e->getMessage());
            return [];
        }
    }
}
