<?php
/**
 * File: helpers/RateLimiter.php
 * Purpose: Rate limiter to enforce Etherscan API limits
 * Free tier: 5 calls/second, 100,000 calls/day
 * Conservative settings: 4 calls/second, 80,000 calls/day
 * Author: MEV Pipeline System
 * Last Modified: 2025-11-15
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/etherscan.php';
require_once __DIR__ . '/Logger.php';

class RateLimiter {
    
    private $pdo;
    private $maxCallsPerSecond;
    private $maxCallsPerDay;
    
    public function __construct() {
        $this->pdo = getDatabaseConnection();
        $this->maxCallsPerSecond = RATE_LIMIT_CALLS_PER_SECOND; // 4 (conservative)
        $this->maxCallsPerDay = RATE_LIMIT_MAX_DAILY_CALLS; // 80,000
    }
    
    /**
     * Check if request is allowed based on rate limits
     * @return bool True if request allowed
     */
    public function allowRequest() {
        // Check per-second limit
        if (!$this->checkPerSecondLimit()) {
            Logger::warning("Per-second rate limit reached", [
                'limit' => $this->maxCallsPerSecond
            ]);
            return false;
        }
        
        // Check daily limit
        if (!$this->checkDailyLimit()) {
            Logger::critical("Daily rate limit reached", [
                'limit' => $this->maxCallsPerDay
            ]);
            return false;
        }
        
        return true;
    }
    
    /**
     * Check per-second rate limit with automatic waiting
     * @param int $maxRetries Maximum number of retry attempts
     * @return bool True if request allowed
     */
    private function checkPerSecondLimit($maxRetries = 5) {
        try {
            $stmt = $this->pdo->prepare("
                SELECT COUNT(*) as count 
                FROM api_call_log 
                WHERE called_at >= DATE_SUB(NOW(), INTERVAL 1 SECOND)
            ");
            $stmt->execute();
            $result = $stmt->fetch();
            
            $currentCalls = $result['count'] ?? 0;
            
            if ($currentCalls >= $this->maxCallsPerSecond) {
                // Wait for the rate limit window to pass
                usleep(250000); // 0.25 seconds
                
                // Recursive check with retry limit
                if ($maxRetries > 0) {
                    return $this->checkPerSecondLimit($maxRetries - 1);
                }
                
                return false;
            }
            
            return true;
            
        } catch (Exception $e) {
            Logger::error("Rate limiter per-second check failed", [
                'error' => $e->getMessage()
            ]);
            // If we can't check, allow request to avoid blocking system
            return true;
        }
    }
    
    /**
     * Check daily rate limit
     * @return bool True if under limit
     */
    private function checkDailyLimit() {
        try {
            $stmt = $this->pdo->prepare("
                SELECT COUNT(*) as count 
                FROM api_call_log 
                WHERE DATE(called_at) = CURDATE()
            ");
            $stmt->execute();
            $result = $stmt->fetch();
            
            $dailyCalls = $result['count'] ?? 0;
            
            if ($dailyCalls >= $this->maxCallsPerDay) {
                Logger::critical("Daily API limit reached", [
                    'calls_today' => $dailyCalls,
                    'limit' => $this->maxCallsPerDay,
                    'date' => date('Y-m-d')
                ]);
                return false;
            }
            
            // Warn at 90% usage
            if ($dailyCalls >= ($this->maxCallsPerDay * 0.9)) {
                Logger::warning("Approaching daily API limit", [
                    'calls_today' => $dailyCalls,
                    'limit' => $this->maxCallsPerDay,
                    'percentage' => round(($dailyCalls / $this->maxCallsPerDay) * 100, 2)
                ]);
            }
            
            return true;
            
        } catch (Exception $e) {
            Logger::error("Rate limiter daily check failed", [
                'error' => $e->getMessage()
            ]);
            return true;
        }
    }
    
    /**
     * Get current rate limit status
     * @return array Status information
     */
    public function getStatus() {
        try {
            // Get today's stats
            $stmt = $this->pdo->prepare("
                SELECT 
                    COUNT(*) as calls_today,
                    MAX(called_at) as last_call,
                    MIN(called_at) as first_call
                FROM api_call_log 
                WHERE DATE(called_at) = CURDATE()
            ");
            $stmt->execute();
            $daily = $stmt->fetch();
            
            // Get last second stats
            $stmt = $this->pdo->prepare("
                SELECT COUNT(*) as calls_last_second
                FROM api_call_log 
                WHERE called_at >= DATE_SUB(NOW(), INTERVAL 1 SECOND)
            ");
            $stmt->execute();
            $recent = $stmt->fetch();
            
            // Get last minute stats
            $stmt = $this->pdo->prepare("
                SELECT COUNT(*) as calls_last_minute
                FROM api_call_log 
                WHERE called_at >= DATE_SUB(NOW(), INTERVAL 1 MINUTE)
            ");
            $stmt->execute();
            $minute = $stmt->fetch();
            
            $callsToday = $daily['calls_today'] ?? 0;
            $callsRemaining = $this->maxCallsPerDay - $callsToday;
            $percentageUsed = $callsToday > 0 ? round(($callsToday / $this->maxCallsPerDay) * 100, 2) : 0;
            
            return [
                'calls_today' => $callsToday,
                'calls_remaining' => $callsRemaining,
                'percentage_used' => $percentageUsed,
                'calls_last_second' => $recent['calls_last_second'] ?? 0,
                'calls_last_minute' => $minute['calls_last_minute'] ?? 0,
                'last_call' => $daily['last_call'] ?? null,
                'first_call' => $daily['first_call'] ?? null,
                'limit_per_second' => $this->maxCallsPerSecond,
                'limit_per_day' => $this->maxCallsPerDay,
                'status' => $callsRemaining > 0 ? 'OK' : 'LIMIT_REACHED'
            ];
            
        } catch (Exception $e) {
            Logger::error("Failed to get rate limit status", [
                'error' => $e->getMessage()
            ]);
            return [
                'error' => $e->getMessage(),
                'status' => 'UNKNOWN'
            ];
        }
    }
    
    /**
     * Get rate limit statistics for specific time period
     * @param string $period Period (hour, day, week)
     * @return array Statistics
     */
    public function getStatistics($period = 'day') {
        try {
            $interval = match($period) {
                'hour' => 'INTERVAL 1 HOUR',
                'day' => 'INTERVAL 1 DAY',
                'week' => 'INTERVAL 7 DAY',
                default => 'INTERVAL 1 DAY'
            };
            
            $stmt = $this->pdo->prepare("
                SELECT 
                    COUNT(*) as total_calls,
                    COUNT(DISTINCT module) as unique_modules,
                    AVG(response_time_ms) as avg_response_time,
                    MAX(response_time_ms) as max_response_time,
                    SUM(CASE WHEN response_status != 200 THEN 1 ELSE 0 END) as error_count,
                    module,
                    COUNT(*) as module_calls
                FROM api_call_log 
                WHERE called_at >= DATE_SUB(NOW(), {$interval})
                GROUP BY module
                ORDER BY module_calls DESC
            ");
            $stmt->execute();
            
            return $stmt->fetchAll();
            
        } catch (Exception $e) {
            Logger::error("Failed to get rate limit statistics", [
                'error' => $e->getMessage(),
                'period' => $period
            ]);
            return [];
        }
    }
    
    /**
     * Reset daily counter (for testing purposes)
     * WARNING: Only use in development/testing
     */
    public function resetDailyCounter() {
        try {
            $stmt = $this->pdo->prepare("DELETE FROM api_call_log WHERE DATE(called_at) = CURDATE()");
            $stmt->execute();
            
            Logger::warning("Daily rate limit counter reset", [
                'rows_deleted' => $stmt->rowCount()
            ]);
            
            return true;
        } catch (Exception $e) {
            Logger::error("Failed to reset daily counter", [
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }
}
