<?php
/**
 * File: helpers/Utils.php
 * Purpose: Utility functions for data validation, conversion, and formatting
 * Author: MEV Pipeline System
 * Last Modified: 2025-11-15
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/etherscan.php';

class Utils {
    
    /**
     * Validate Ethereum address format
     * @param string $address Ethereum address
     * @return bool True if valid
     */
    public static function isValidAddress($address) {
        return preg_match('/^0x[a-fA-F0-9]{40}$/', $address) === 1;
    }
    
    /**
     * Validate transaction hash format
     * @param string $hash Transaction hash
     * @return bool True if valid
     */
    public static function isValidTxHash($hash) {
        return preg_match('/^0x[a-fA-F0-9]{64}$/', $hash) === 1;
    }
    
    /**
     * Convert wei to ether
     * @param string $wei Amount in wei
     * @return string Amount in ether
     */
    public static function weiToEther($wei) {
        return bcdiv($wei, '1000000000000000000', 18);
    }
    
    /**
     * Convert ether to wei
     * @param string $ether Amount in ether
     * @return string Amount in wei
     */
    public static function etherToWei($ether) {
        return bcmul($ether, '1000000000000000000', 0);
    }
    
    /**
     * Convert token amount from raw value based on decimals
     * @param string $rawValue Raw token value
     * @param int $decimals Token decimals
     * @return string Formatted token amount
     */
    public static function formatTokenAmount($rawValue, $decimals = 18) {
        $divisor = bcpow('10', (string)$decimals, 0);
        return bcdiv($rawValue, $divisor, $decimals);
    }
    
    /**
     * Get token decimals from config
     * @param string $tokenAddress Token contract address
     * @return int Token decimals (default 18)
     */
    public static function getTokenDecimals($tokenAddress) {
        $decimals = TOKEN_DECIMALS[strtolower($tokenAddress)] ?? null;
        return $decimals !== null ? $decimals : 18;
    }
    
    /**
     * Get token symbol from address
     * @param string $tokenAddress Token contract address
     * @return string Token symbol
     */
    public static function getTokenSymbol($tokenAddress) {
        $tokens = array_flip(SUPPORTED_TOKENS);
        return $tokens[strtolower($tokenAddress)] ?? 'UNKNOWN';
    }
    
    /**
     * Get approximate USD value for token amount
     * @param string $tokenAddress Token contract address
     * @param string $amount Token amount (formatted, not raw)
     * @return float USD value
     */
    public static function getApproximateUsdValue($tokenAddress, $amount) {
        $tokenAddress = strtolower($tokenAddress);
        $price = APPROXIMATE_PRICES[$tokenAddress] ?? 0;
        return bcmul($amount, (string)$price, 2);
    }
    
    /**
     * Convert gwei to wei
     * @param string $gwei Amount in gwei
     * @return string Amount in wei
     */
    public static function gweiToWei($gwei) {
        return bcmul($gwei, '1000000000', 0);
    }
    
    /**
     * Convert wei to gwei
     * @param string $wei Amount in wei
     * @return string Amount in gwei
     */
    public static function weiToGwei($wei) {
        return bcdiv($wei, '1000000000', 9);
    }
    
    /**
     * Calculate gas cost in ETH
     * @param int $gasUsed Gas used
     * @param string $gasPriceWei Gas price in wei
     * @return string Cost in ETH
     */
    public static function calculateGasCostEth($gasUsed, $gasPriceWei) {
        $costWei = bcmul((string)$gasUsed, $gasPriceWei, 0);
        return self::weiToEther($costWei);
    }
    
    /**
     * Calculate gas cost in USD
     * @param int $gasUsed Gas used
     * @param string $gasPriceWei Gas price in wei
     * @param float $ethPriceUsd ETH price in USD
     * @return float Cost in USD
     */
    public static function calculateGasCostUsd($gasUsed, $gasPriceWei, $ethPriceUsd) {
        $costEth = self::calculateGasCostEth($gasUsed, $gasPriceWei);
        return bcmul($costEth, (string)$ethPriceUsd, 2);
    }
    
    /**
     * Get DEX name from router address
     * @param string $address Router address
     * @return string DEX name
     */
    public static function getDexName($address) {
        $address = strtolower($address);
        
        foreach (MAJOR_DEX_CONTRACTS as $name => $contracts) {
            if (strtolower($contracts['router']) === $address || 
                strtolower($contracts['factory']) === $address) {
                return str_replace('_', ' ', ucwords(strtolower($name), '_'));
            }
        }
        
        return 'Unknown DEX';
    }
    
    /**
     * Get swap event signature for DEX
     * @param string $dexName DEX name
     * @return string Event signature (topic0)
     */
    public static function getSwapSignature($dexName) {
        $dexKey = strtoupper(str_replace(' ', '_', $dexName));
        return MAJOR_DEX_CONTRACTS[$dexKey]['swap_signature'] ?? EVENT_SIGNATURES['UNISWAP_V2_SWAP'];
    }
    
    /**
     * Sanitize and normalize Ethereum address
     * @param string $address Ethereum address
     * @return string Normalized address (checksummed)
     */
    public static function normalizeAddress($address) {
        // Remove whitespace and convert to lowercase
        $address = strtolower(trim($address));
        
        // Ensure 0x prefix
        if (substr($address, 0, 2) !== '0x') {
            $address = '0x' . $address;
        }
        
        return $address;
    }
    
    /**
     * Calculate percentage difference between two values
     * @param float $value1 First value
     * @param float $value2 Second value
     * @return float Percentage difference
     */
    public static function percentageDifference($value1, $value2) {
        if ($value2 == 0) {
            return 0;
        }
        return (($value1 - $value2) / $value2) * 100;
    }
    
    /**
     * Format large numbers with suffixes (K, M, B)
     * @param float $number Number to format
     * @param int $decimals Decimal places
     * @return string Formatted number
     */
    public static function formatLargeNumber($number, $decimals = 2) {
        if ($number >= 1000000000) {
            return round($number / 1000000000, $decimals) . 'B';
        } elseif ($number >= 1000000) {
            return round($number / 1000000, $decimals) . 'M';
        } elseif ($number >= 1000) {
            return round($number / 1000, $decimals) . 'K';
        }
        return round($number, $decimals);
    }
    
    /**
     * Get time ago string from timestamp
     * @param string $timestamp Timestamp
     * @return string Time ago string
     */
    public static function timeAgo($timestamp) {
        $time = strtotime($timestamp);
        $diff = time() - $time;
        
        if ($diff < 60) {
            return $diff . ' seconds ago';
        } elseif ($diff < 3600) {
            return floor($diff / 60) . ' minutes ago';
        } elseif ($diff < 86400) {
            return floor($diff / 3600) . ' hours ago';
        } else {
            return floor($diff / 86400) . ' days ago';
        }
    }
    
    /**
     * Generate unique ID for opportunity
     * @param string $type Opportunity type
     * @return string Unique ID
     */
    public static function generateOpportunityId($type) {
        return strtoupper($type) . '_' . date('YmdHis') . '_' . substr(md5(uniqid()), 0, 8);
    }
    
    /**
     * Decode hex string to UTF-8
     * @param string $hex Hex string
     * @return string Decoded string
     */
    public static function hexToString($hex) {
        $hex = str_replace('0x', '', $hex);
        $string = '';
        for ($i = 0; $i < strlen($hex) - 1; $i += 2) {
            $string .= chr(hexdec($hex[$i] . $hex[$i + 1]));
        }
        return $string;
    }
    
    /**
     * Convert hex to decimal
     * @param string $hex Hex string
     * @return string Decimal string
     */
    public static function hexToDec($hex) {
        $hex = str_replace('0x', '', $hex);
        return (string)hexdec($hex);
    }
    
    /**
     * Convert decimal to hex
     * @param int|string $dec Decimal value
     * @return string Hex string with 0x prefix
     */
    public static function decToHex($dec) {
        return '0x' . dechex((int)$dec);
    }
    
    /**
     * Check if system is in maintenance mode
     * @return bool True if in maintenance mode
     */
    public static function isMaintenanceMode() {
        try {
            return (bool)getConfigValue('maintenance_mode', false);
        } catch (Exception $e) {
            return false;
        }
    }
    
    /**
     * Get current ETH price (from cache or config)
     * @return float ETH price in USD
     */
    public static function getEthPrice() {
        // In production, this should fetch from a price oracle
        // For now, return approximate price from config
        return APPROXIMATE_PRICES['0xC02aaA39b223FE8D0A0e5C4F27eAD9083C756Cc2'] ?? 3500.00;
    }
    
    /**
     * Truncate address for display
     * @param string $address Ethereum address
     * @param int $start Characters to show at start
     * @param int $end Characters to show at end
     * @return string Truncated address
     */
    public static function truncateAddress($address, $start = 6, $end = 4) {
        if (strlen($address) <= ($start + $end)) {
            return $address;
        }
        return substr($address, 0, $start) . '...' . substr($address, -$end);
    }
    
    /**
     * Validate and sanitize input parameters
     * @param array $params Parameters to validate
     * @param array $rules Validation rules
     * @return array Sanitized parameters
     * @throws Exception if validation fails
     */
    public static function validateParams($params, $rules) {
        $sanitized = [];
        
        foreach ($rules as $key => $rule) {
            $value = $params[$key] ?? null;
            
            // Check required fields
            if (isset($rule['required']) && $rule['required'] && empty($value)) {
                throw new Exception("Missing required parameter: {$key}");
            }
            
            // Type validation
            if (isset($rule['type']) && !empty($value)) {
                switch ($rule['type']) {
                    case 'address':
                        if (!self::isValidAddress($value)) {
                            throw new Exception("Invalid Ethereum address: {$key}");
                        }
                        $value = self::normalizeAddress($value);
                        break;
                    case 'int':
                        $value = (int)$value;
                        break;
                    case 'float':
                        $value = (float)$value;
                        break;
                    case 'string':
                        $value = trim($value);
                        break;
                }
            }
            
            // Set default if provided
            if (empty($value) && isset($rule['default'])) {
                $value = $rule['default'];
            }
            
            $sanitized[$key] = $value;
        }
        
        return $sanitized;
    }
}
