<?php
/**
 * File: processors/detect_arbitrage.php
 * Purpose: Detect cross-DEX arbitrage opportunities
 * Called by: Cron every 3 minutes
 * Author: MEV Pipeline System
 * Last Modified: 2025-11-15
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../helpers/Logger.php';
require_once __DIR__ . '/../helpers/Utils.php';

/**
 * Detect arbitrage opportunities
 * @return bool Success status
 */
function detectArbitrage() {
    try {
        Logger::info("Starting arbitrage detection");
        
        $pdo = getDatabaseConnection();
        
        // Get recent swaps (last 5 minutes)
        $stmt = $pdo->query("
            SELECT 
                dex_name,
                token_in,
                token_out,
                amount_in_usd,
                amount_out_usd,
                tx_hash,
                timestamp
            FROM dex_swaps
            WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 5 MINUTE)
                AND amount_in_usd IS NOT NULL
                AND amount_out_usd IS NOT NULL
            ORDER BY timestamp DESC
        ");
        
        $swaps = $stmt->fetchAll();
        
        if (empty($swaps)) {
            Logger::debug("No recent swaps for arbitrage detection");
            return true;
        }
        
        // Group swaps by token pair
        $swapsByPair = groupSwapsByTokenPair($swaps);
        
        $opportunitiesFound = 0;
        
        // Analyze each token pair across DEXes
        foreach ($swapsByPair as $pair => $pairSwaps) {
            $opportunities = findArbitrageInPair($pairSwaps);
            
            foreach ($opportunities as $opportunity) {
                if (saveOpportunity($opportunity)) {
                    $opportunitiesFound++;
                }
            }
        }
        
        Logger::info("Arbitrage detection completed", [
            'opportunities_found' => $opportunitiesFound,
            'swaps_analyzed' => count($swaps)
        ]);
        
        return true;
        
    } catch (Exception $e) {
        Logger::error("Arbitrage detection failed", [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
        return false;
    }
}

/**
 * Group swaps by token pair
 * @param array $swaps Array of swap records
 * @return array Grouped swaps
 */
function groupSwapsByTokenPair($swaps) {
    $grouped = [];
    
    foreach ($swaps as $swap) {
        $tokenIn = $swap['token_in'];
        $tokenOut = $swap['token_out'];
        
        // Create normalized pair key (sort addresses to group A-B and B-A together)
        $addresses = [$tokenIn, $tokenOut];
        sort($addresses);
        $pairKey = implode('_', $addresses);
        
        if (!isset($grouped[$pairKey])) {
            $grouped[$pairKey] = [];
        }
        
        $grouped[$pairKey][] = $swap;
    }
    
    return $grouped;
}

/**
 * Find arbitrage opportunities in a token pair
 * @param array $pairSwaps Swaps for this pair
 * @return array Arbitrage opportunities
 */
function findArbitrageInPair($pairSwaps) {
    $opportunities = [];
    
    // Group by DEX
    $swapsByDex = [];
    foreach ($pairSwaps as $swap) {
        $dex = $swap['dex_name'];
        if (!isset($swapsByDex[$dex])) {
            $swapsByDex[$dex] = [];
        }
        $swapsByDex[$dex][] = $swap;
    }
    
    // Need at least 2 DEXes for arbitrage
    if (count($swapsByDex) < 2) {
        return $opportunities;
    }
    
    // Compare prices across DEXes
    $dexPrices = [];
    foreach ($swapsByDex as $dex => $swaps) {
        // Calculate average price for this DEX
        $totalIn = 0;
        $totalOut = 0;
        
        foreach ($swaps as $swap) {
            $totalIn += $swap['amount_in_usd'];
            $totalOut += $swap['amount_out_usd'];
        }
        
        if ($totalIn > 0) {
            $dexPrices[$dex] = $totalOut / $totalIn;
        }
    }
    
    // Find price differences
    foreach ($dexPrices as $dex1 => $price1) {
        foreach ($dexPrices as $dex2 => $price2) {
            if ($dex1 === $dex2) continue;
            
            $priceDiff = abs($price1 - $price2);
            $percentDiff = ($priceDiff / max($price1, $price2)) * 100;
            
            // Check if profitable (above minimum threshold)
            if ($percentDiff >= MEV_MIN_PROFIT_PERCENTAGE) {
                
                // Get current gas price
                $gasPrice = getCurrentGasPrice();
                
                // Estimate gas cost (simplified - use 200k gas for arbitrage)
                $estimatedGas = 200000;
                $ethPrice = Utils::getEthPrice();
                $gasCostUsd = Utils::calculateGasCostUsd($estimatedGas, $gasPrice, $ethPrice);
                
                // Calculate potential profit (simplified)
                $tradeSize = 1000; // $1000 example trade
                $grossProfit = ($percentDiff / 100) * $tradeSize;
                $netProfit = $grossProfit - $gasCostUsd;
                
                // Only create opportunity if net profit is positive
                if ($netProfit > 0) {
                    $confidence = calculateConfidenceScore($percentDiff, count($pairSwaps));
                    
                    $opportunities[] = [
                        'opportunity_type' => 'ARBITRAGE',
                        'token_pair' => $pairSwaps[0]['token_in'] . '/' . $pairSwaps[0]['token_out'],
                        'dex_a' => $dex1,
                        'dex_b' => $dex2,
                        'price_diff_percent' => $percentDiff,
                        'expected_profit_usd' => $grossProfit,
                        'gas_cost_usd' => $gasCostUsd,
                        'net_profit_usd' => $netProfit,
                        'profit_percentage' => ($netProfit / $tradeSize) * 100,
                        'confidence_score' => $confidence,
                        'urgency_level' => determineUrgency($netProfit, $percentDiff),
                        'related_swaps' => $pairSwaps
                    ];
                }
            }
        }
    }
    
    return $opportunities;
}

/**
 * Calculate confidence score based on various factors
 * @param float $priceDiff Price difference percentage
 * @param int $sampleSize Number of swaps in sample
 * @return float Confidence score (0-1)
 */
function calculateConfidenceScore($priceDiff, $sampleSize) {
    // Base confidence on price difference
    $baseConfidence = min($priceDiff / 10, 0.8); // Max 0.8 from price diff
    
    // Adjust for sample size
    $sampleConfidence = min($sampleSize / 10, 0.2); // Max 0.2 from sample size
    
    return round($baseConfidence + $sampleConfidence, 2);
}

/**
 * Determine urgency level
 * @param float $netProfit Net profit in USD
 * @param float $priceDiff Price difference percentage
 * @return string Urgency level
 */
function determineUrgency($netProfit, $priceDiff) {
    if ($netProfit > 100 || $priceDiff > 5) {
        return 'HIGH';
    } elseif ($netProfit > 20 || $priceDiff > 2) {
        return 'MEDIUM';
    }
    return 'LOW';
}

/**
 * Get current gas price in wei
 * @return string Gas price in wei
 */
function getCurrentGasPrice() {
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->query("
            SELECT fast_gas_price 
            FROM gas_tracker 
            ORDER BY timestamp DESC 
            LIMIT 1
        ");
        $result = $stmt->fetch();
        
        $gasPriceGwei = $result['fast_gas_price'] ?? 50;
        return Utils::gweiToWei($gasPriceGwei);
        
    } catch (Exception $e) {
        // Default to 50 gwei if error
        return Utils::gweiToWei(50);
    }
}

/**
 * Save opportunity to database
 * @param array $opportunity Opportunity data
 * @return bool Success status
 */
function saveOpportunity($opportunity) {
    try {
        $pdo = getDatabaseConnection();
        
        // Check if similar opportunity already exists (avoid duplicates)
        $checkStmt = $pdo->prepare("
            SELECT id FROM mev_opportunities
            WHERE opportunity_type = 'ARBITRAGE'
                AND status = 'pending'
                AND expires_at > NOW()
                AND strategy_data LIKE ?
        ");
        $checkStmt->execute(['%' . $opportunity['token_pair'] . '%']);
        
        if ($checkStmt->fetch()) {
            return false; // Similar opportunity already exists
        }
        
        // Prepare strategy data
        $strategyData = json_encode([
            'token_pair' => $opportunity['token_pair'],
            'dex_a' => $opportunity['dex_a'],
            'dex_b' => $opportunity['dex_b'],
            'price_diff' => $opportunity['price_diff_percent']
        ]);
        
        // Insert opportunity
        $stmt = $pdo->prepare("
            INSERT INTO mev_opportunities
            (opportunity_type, status, expected_profit_usd, gas_cost_usd, net_profit_usd,
             profit_percentage, confidence_score, urgency_level, strategy_data,
             detected_at, expires_at)
            VALUES (?, 'pending', ?, ?, ?, ?, ?, ?, ?, NOW(), DATE_ADD(NOW(), INTERVAL ? MINUTE))
        ");
        
        $stmt->execute([
            $opportunity['opportunity_type'],
            $opportunity['expected_profit_usd'],
            $opportunity['gas_cost_usd'],
            $opportunity['net_profit_usd'],
            $opportunity['profit_percentage'],
            $opportunity['confidence_score'],
            $opportunity['urgency_level'],
            $strategyData,
            OPPORTUNITY_EXPIRY_MINUTES
        ]);
        
        Logger::info("Arbitrage opportunity detected", [
            'net_profit' => number_format($opportunity['net_profit_usd'], 2),
            'token_pair' => $opportunity['token_pair'],
            'dex_a' => $opportunity['dex_a'],
            'dex_b' => $opportunity['dex_b'],
            'urgency' => $opportunity['urgency_level']
        ]);
        
        return true;
        
    } catch (Exception $e) {
        Logger::error("Failed to save arbitrage opportunity", [
            'error' => $e->getMessage()
        ]);
        return false;
    }
}

// CLI execution
if (php_sapi_name() === 'cli') {
    $testMode = false;
    
    if (isset($argv) && in_array('--test', $argv)) {
        $testMode = true;
        echo "========================================\n";
        echo "Arbitrage Detector - Test Mode\n";
        echo "========================================\n\n";
    }
    
    $success = detectArbitrage();
    
    if ($testMode) {
        echo $success ? "✓ Arbitrage detection completed\n" : "✗ Detection failed\n";
        echo "\nTest complete.\n";
    }
    
    exit($success ? 0 : 1);
}
