<?php
/**
 * File: processors/detect_sandwich.php
 * Purpose: Detect large swaps vulnerable to sandwich attacks
 * Called by: Cron every 3 minutes
 * Author: MEV Pipeline System
 * Last Modified: 2025-11-15
 */

error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../helpers/Logger.php';
require_once __DIR__ . '/../helpers/Utils.php';

/**
 * Detect sandwich attack opportunities
 * @return bool Success status
 */
function detectSandwich() {
    try {
        Logger::info("Starting sandwich detection");
        
        $pdo = getDatabaseConnection();
        
        // Get recent large swaps (potential sandwich targets)
        $stmt = $pdo->query("
            SELECT 
                id,
                tx_hash,
                dex_name,
                token_in,
                token_out,
                amount_in_usd,
                amount_out_usd,
                price_impact,
                timestamp
            FROM dex_swaps
            WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 10 MINUTE)
                AND is_large_swap = 1
                AND amount_in_usd >= " . LARGE_SWAP_THRESHOLD_USD . "
            ORDER BY timestamp DESC
        ");
        
        $largeSwaps = $stmt->fetchAll();
        
        if (empty($largeSwaps)) {
            Logger::debug("No large swaps for sandwich detection");
            return true;
        }
        
        $opportunitiesFound = 0;
        
        foreach ($largeSwaps as $swap) {
            $opportunity = analyzeSandwichOpportunity($swap);
            
            if ($opportunity && saveOpportunity($opportunity)) {
                $opportunitiesFound++;
            }
        }
        
        Logger::info("Sandwich detection completed", [
            'opportunities_found' => $opportunitiesFound,
            'large_swaps_analyzed' => count($largeSwaps)
        ]);
        
        return true;
        
    } catch (Exception $e) {
        Logger::error("Sandwich detection failed", [
            'error' => $e->getMessage()
        ]);
        return false;
    }
}

/**
 * Analyze sandwich opportunity for a large swap
 * @param array $swap Large swap data
 * @return array|null Opportunity data or null
 */
function analyzeSandwichOpportunity($swap) {
    try {
        // Estimate price impact (simplified calculation)
        $swapSize = $swap['amount_in_usd'];
        $estimatedPriceImpact = estimatePriceImpact($swapSize);
        
        if ($estimatedPriceImpact < 1) {
            return null; // Not enough price impact for profitable sandwich
        }
        
        // Calculate potential profits
        $frontRunSize = min($swapSize * 0.5, 10000); // Front-run with up to $10k
        $frontRunProfit = ($estimatedPriceImpact / 100) * $frontRunSize;
        
        $backRunSize = $frontRunSize;
        $backRunProfit = ($estimatedPriceImpact / 100) * $backRunSize;
        
        $totalGrossProfit = $frontRunProfit + $backRunProfit;
        
        // Calculate gas costs (3 transactions: front-run, victim, back-run)
        $gasPrice = getCurrentGasPrice();
        $ethPrice = Utils::getEthPrice();
        
        // Estimate gas usage (front-run + back-run)
        $frontRunGas = 150000;
        $backRunGas = 150000;
        $totalGas = $frontRunGas + $backRunGas;
        
        $gasCostUsd = Utils::calculateGasCostUsd($totalGas, $gasPrice, $ethPrice);
        
        // Calculate net profit
        $netProfit = $totalGrossProfit - $gasCostUsd;
        
        // Only create opportunity if profitable
        if ($netProfit <= 0) {
            return null;
        }
        
        // Check if profit meets minimum threshold
        $profitPercentage = ($netProfit / ($frontRunSize + $backRunSize)) * 100;
        
        if ($profitPercentage < MEV_MIN_PROFIT_PERCENTAGE) {
            return null;
        }
        
        // Calculate confidence score
        $confidence = calculateConfidenceScore($estimatedPriceImpact, $swapSize);
        
        if ($confidence < MEV_CONFIDENCE_THRESHOLD) {
            return null;
        }
        
        // Build opportunity data
        return [
            'opportunity_type' => 'SANDWICH',
            'target_tx_hash' => $swap['tx_hash'],
            'dex_name' => $swap['dex_name'],
            'target_swap_size' => $swapSize,
            'estimated_price_impact' => $estimatedPriceImpact,
            'front_run_size' => $frontRunSize,
            'back_run_size' => $backRunSize,
            'expected_profit_usd' => $totalGrossProfit,
            'gas_cost_usd' => $gasCostUsd,
            'net_profit_usd' => $netProfit,
            'profit_percentage' => $profitPercentage,
            'confidence_score' => $confidence,
            'urgency_level' => 'HIGH', // Sandwich opportunities are time-sensitive
            'token_in' => $swap['token_in'],
            'token_out' => $swap['token_out']
        ];
        
    } catch (Exception $e) {
        Logger::error("Failed to analyze sandwich opportunity", [
            'tx_hash' => $swap['tx_hash'] ?? 'unknown',
            'error' => $e->getMessage()
        ]);
        return null;
    }
}

/**
 * Estimate price impact based on swap size
 * @param float $swapSizeUsd Swap size in USD
 * @return float Estimated price impact percentage
 */
function estimatePriceImpact($swapSizeUsd) {
    // Simplified price impact estimation
    // In production, use actual pool liquidity data
    
    if ($swapSizeUsd >= 1000000) {
        return 5.0; // 5% impact for $1M+ swaps
    } elseif ($swapSizeUsd >= 500000) {
        return 3.0; // 3% impact for $500k+ swaps
    } elseif ($swapSizeUsd >= 100000) {
        return 2.0; // 2% impact for $100k+ swaps
    } elseif ($swapSizeUsd >= 50000) {
        return 1.0; // 1% impact for $50k+ swaps
    }
    
    return 0.5; // 0.5% impact for smaller swaps
}

/**
 * Calculate confidence score
 * @param float $priceImpact Price impact percentage
 * @param float $swapSize Swap size in USD
 * @return float Confidence score (0-1)
 */
function calculateConfidenceScore($priceImpact, $swapSize) {
    // Higher price impact = higher confidence
    $impactScore = min($priceImpact / 5, 0.6);
    
    // Larger swaps = higher confidence
    $sizeScore = min($swapSize / 1000000, 0.4);
    
    return round($impactScore + $sizeScore, 2);
}

/**
 * Get current gas price in wei
 * @return string Gas price in wei
 */
function getCurrentGasPrice() {
    try {
        $pdo = getDatabaseConnection();
        $stmt = $pdo->query("
            SELECT fast_gas_price 
            FROM gas_tracker 
            ORDER BY timestamp DESC 
            LIMIT 1
        ");
        $result = $stmt->fetch();
        
        $gasPriceGwei = $result['fast_gas_price'] ?? 50;
        return Utils::gweiToWei($gasPriceGwei);
        
    } catch (Exception $e) {
        return Utils::gweiToWei(50);
    }
}

/**
 * Save sandwich opportunity to database
 * @param array $opportunity Opportunity data
 * @return bool Success status
 */
function saveOpportunity($opportunity) {
    try {
        $pdo = getDatabaseConnection();
        
        // Check if opportunity for this tx already exists
        $checkStmt = $pdo->prepare("
            SELECT id FROM mev_opportunities
            WHERE opportunity_type = 'SANDWICH'
                AND status = 'pending'
                AND related_tx_hashes LIKE ?
        ");
        $checkStmt->execute(['%' . $opportunity['target_tx_hash'] . '%']);
        
        if ($checkStmt->fetch()) {
            return false; // Already detected
        }
        
        // Prepare data
        $tokenAddresses = json_encode([
            $opportunity['token_in'],
            $opportunity['token_out']
        ]);
        
        $strategyData = json_encode([
            'target_tx' => $opportunity['target_tx_hash'],
            'dex' => $opportunity['dex_name'],
            'price_impact' => $opportunity['estimated_price_impact'],
            'front_run_size' => $opportunity['front_run_size'],
            'back_run_size' => $opportunity['back_run_size']
        ]);
        
        $relatedTxHashes = json_encode([$opportunity['target_tx_hash']]);
        
        $executionSteps = json_encode([
            [
                'step' => 1,
                'action' => 'FRONT_RUN',
                'description' => 'Buy before target transaction',
                'amount_usd' => $opportunity['front_run_size']
            ],
            [
                'step' => 2,
                'action' => 'WAIT',
                'description' => 'Wait for target transaction to execute'
            ],
            [
                'step' => 3,
                'action' => 'BACK_RUN',
                'description' => 'Sell after target transaction',
                'amount_usd' => $opportunity['back_run_size']
            ]
        ]);
        
        // Insert opportunity
        $stmt = $pdo->prepare("
            INSERT INTO mev_opportunities
            (opportunity_type, status, token_addresses, expected_profit_usd, gas_cost_usd,
             net_profit_usd, profit_percentage, confidence_score, urgency_level,
             related_tx_hashes, strategy_data, execution_steps, detected_at, expires_at)
            VALUES (?, 'pending', ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), DATE_ADD(NOW(), INTERVAL 2 MINUTE))
        ");
        
        $stmt->execute([
            $opportunity['opportunity_type'],
            $tokenAddresses,
            $opportunity['expected_profit_usd'],
            $opportunity['gas_cost_usd'],
            $opportunity['net_profit_usd'],
            $opportunity['profit_percentage'],
            $opportunity['confidence_score'],
            $opportunity['urgency_level'],
            $relatedTxHashes,
            $strategyData,
            $executionSteps
        ]);
        
        Logger::info("Sandwich opportunity detected", [
            'target_tx' => Utils::truncateAddress($opportunity['target_tx_hash']),
            'net_profit' => number_format($opportunity['net_profit_usd'], 2),
            'price_impact' => $opportunity['estimated_price_impact'] . '%',
            'confidence' => $opportunity['confidence_score']
        ]);
        
        return true;
        
    } catch (Exception $e) {
        Logger::error("Failed to save sandwich opportunity", [
            'error' => $e->getMessage()
        ]);
        return false;
    }
}

// CLI execution
if (php_sapi_name() === 'cli') {
    $testMode = false;
    
    if (isset($argv) && in_array('--test', $argv)) {
        $testMode = true;
        echo "========================================\n";
        echo "Sandwich Detector - Test Mode\n";
        echo "========================================\n\n";
    }
    
    $success = detectSandwich();
    
    if ($testMode) {
        echo $success ? "✓ Sandwich detection completed\n" : "✗ Detection failed\n";
        echo "\nTest complete.\n";
    }
    
    exit($success ? 0 : 1);
}
