# MEV Data Pipeline - Installation Guide

## System Overview
Production-ready Ethereum MEV (Maximal Extractable Value) data collection pipeline that replaces simulated opportunity engines with real blockchain data from Etherscan API.

**Live System**: `https://core.poipoe.com/mevdata/`  
**Frontend**: `https://app.poipoe.com`

---

## Prerequisites

### Server Requirements
- **Hosting**: cPanel hosting (qhoster)
- **PHP**: 7.4 or higher
- **Database**: MariaDB/MySQL
- **Cron Access**: Yes (required)
- **SSL**: Recommended

### API Requirements
- **Etherscan API Key**: `ZR2CAV375V5R3GCHWYVNZNBZZGTNZ5KH78`
- **Rate Limits**: 5 calls/second, 100,000 calls/day (free tier)
- **Conservative Settings**: 4 calls/second, 80,000 calls/day (configured)

---

## Installation Steps

### 1. Upload Files

Upload all files to your server at:
```
/home/poipoeco/public_html/core.poipoe.com/mevdata/
```

Ensure the following directory structure:
```
mevdata/
├── config/
├── collectors/
├── processors/
├── api/
├── helpers/
├── cron/
├── admin/
├── setup/
├── logs/
└── cache/
```

### 2. Set Directory Permissions

```bash
chmod 755 mevdata/
chmod 755 mevdata/*/
chmod 777 mevdata/logs/
chmod 777 mevdata/cache/
```

### 3. Database Setup

**Database Credentials:**
- Database Name: `poipoeco_mevdata`
- Username: `poipoeco_mevdata_user`
- Password: `32TbgulnE7YOd2G0e5`
- Host: `localhost`

**Import Schema:**
```bash
mysql -u poipoeco_mevdata_user -p poipoeco_mevdata < setup/install.sql
```

Or via phpMyAdmin:
1. Login to phpMyAdmin
2. Select database `poipoeco_mevdata`
3. Go to Import tab
4. Choose file `setup/install.sql`
5. Click "Go"

**Verify Installation:**
```sql
USE poipoeco_mevdata;
SHOW TABLES;
-- Should show 9 tables
SELECT COUNT(*) FROM watched_addresses;
-- Should show 6 watched addresses
```

### 4. Configure Settings

Edit `config/database.php` if credentials differ:
```php
define('DB_HOST', 'localhost');
define('DB_NAME', 'poipoeco_mevdata');
define('DB_USER', 'poipoeco_mevdata_user');
define('DB_PASS', '32TbgulnE7YOd2G0e5');
```

Edit `config/etherscan.php` if API key differs:
```php
define('ETHERSCAN_API_KEY', 'ZR2CAV375V5R3GCHWYVNZNBZZGTNZ5KH78');
```

### 5. Test Database Connection

```bash
php -r "require 'config/database.php'; testDatabaseConnection() ? print('OK\n') : print('FAILED\n');"
```

### 6. Test Collectors (Manual)

```bash
# Test gas tracker
php collectors/collect_gas_tracker.php --test

# Test token transfers
php collectors/collect_token_transfers.php --test

# Test DEX swaps
php collectors/collect_dex_swaps.php --test

# Test whale movements
php collectors/collect_whale_movements.php --test
```

### 7. Configure Cron Jobs

**Important**: These cron jobs should already be configured in your cPanel. Verify they exist:

```bash
# Every minute - Gas Tracker
* * * * * /usr/local/bin/php /home/poipoeco/public_html/core.poipoe.com/mevdata/cron/cron_master.php --task=gas_tracker

# Every minute - DEX Events
* * * * * /usr/local/bin/php /home/poipoeco/public_html/core.poipoe.com/mevdata/cron/cron_master.php --task=dex_events

# Every 2 minutes - Whale Movements
*/2 * * * * /usr/local/bin/php /home/poipoeco/public_html/core.poipoe.com/mevdata/cron/cron_master.php --task=whale_movements

# Every 2 minutes - Token Transfers
*/2 * * * * /usr/local/bin/php /home/poipoeco/public_html/core.poipoe.com/mevdata/cron/cron_master.php --task=token_transfers

# Every 3 minutes - MEV Detection
*/3 * * * * /usr/local/bin/php /home/poipoeco/public_html/core.poipoe.com/mevdata/cron/cron_master.php --task=detect_opportunities

# Twice per hour - Cleanup (at :00 and :30)
0,30 * * * * /usr/local/bin/php /home/poipoeco/public_html/core.poipoe.com/mevdata/cron/cron_master.php --task=cleanup
```

**To add/verify in cPanel:**
1. Login to cPanel
2. Go to "Cron Jobs"
3. Add each command above with appropriate timing

### 8. Verify Cron Jobs Are Running

Wait 2-3 minutes after setup, then check:

```bash
# Check logs
tail -f logs/app_$(date +%Y-%m-%d).log

# Check database
mysql -u poipoeco_mevdata_user -p poipoeco_mevdata -e "SELECT COUNT(*) FROM gas_tracker;"
mysql -u poipoeco_mevdata_user -p poipoeco_mevdata -e "SELECT COUNT(*) FROM api_call_log;"
```

### 9. Test API Endpoints

**Health Check:**
```bash
php cron/cron_master.php --action=health
```

Expected output:
```json
{
    "overall_health": "HEALTHY",
    "database_connection": "OK",
    "api_key_valid": true,
    ...
}
```

**Test API Endpoints (from browser or curl):**
```bash
# Get opportunities
curl "https://core.poipoe.com/mevdata/api/get_opportunities.php?limit=5"

# Get live data
curl "https://core.poipoe.com/mevdata/api/get_live_data.php?type=all"
```

### 10. Frontend Integration

The frontend at `https://app.poipoe.com` should make requests to:

```javascript
// Get MEV opportunities
fetch('https://core.poipoe.com/mevdata/api/get_opportunities.php?status=pending&limit=50')

// Get live market data
fetch('https://core.poipoe.com/mevdata/api/get_live_data.php?type=all')
```

CORS is already configured to allow requests from `https://app.poipoe.com`.

---

## Monitoring & Maintenance

### Daily Checks

1. **Check Health Status:**
   ```bash
   php cron/cron_master.php --action=health
   ```

2. **Check API Usage:**
   ```sql
   SELECT COUNT(*) as calls_today FROM api_call_log WHERE DATE(called_at) = CURDATE();
   ```
   Should be under 80,000 calls/day.

3. **Check for Errors:**
   ```sql
   SELECT * FROM error_log WHERE created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR) ORDER BY created_at DESC LIMIT 10;
   ```

4. **Check Opportunities:**
   ```sql
   SELECT COUNT(*) as active_opportunities FROM mev_opportunities WHERE status = 'pending' AND expires_at > NOW();
   ```

### Log Files

Logs are stored in `logs/` directory:
- Format: `app_YYYY-MM-DD.log`
- Rotation: Daily, max 10MB per file
- Retention: 30 days (auto-cleanup)

**View logs:**
```bash
tail -f logs/app_$(date +%Y-%m-%d).log
```

### Database Maintenance

**Weekly:**
```sql
-- Optimize tables
OPTIMIZE TABLE gas_tracker, token_transfers, dex_swaps, mev_opportunities;
```

**Monthly:**
```sql
-- Check database size
SELECT 
    table_name,
    ROUND(((data_length + index_length) / 1024 / 1024), 2) AS `Size (MB)`
FROM information_schema.TABLES
WHERE table_schema = 'poipoeco_mevdata'
ORDER BY (data_length + index_length) DESC;
```

---

## Troubleshooting

### Issue: No data being collected

**Check:**
1. Cron jobs are running: `crontab -l`
2. Database connection: `php -r "require 'config/database.php'; testDatabaseConnection();"`
3. API key valid: Check Etherscan dashboard
4. Check logs: `tail -f logs/app_*.log`

### Issue: API rate limit exceeded

**Solution:**
```sql
-- Check current usage
SELECT COUNT(*) FROM api_call_log WHERE DATE(called_at) = CURDATE();

-- If needed, reduce collection frequency in cron jobs
```

### Issue: Database connection errors

**Check:**
1. Database credentials in `config/database.php`
2. Database user has proper permissions
3. MySQL service is running

### Issue: Cron jobs not executing

**Check:**
1. PHP path: `which php` (should be `/usr/local/bin/php`)
2. File permissions: `ls -la cron/cron_master.php`
3. Cron logs: Check cPanel cron email notifications

---

## Performance Optimization

### Recommended Indexes (already created)

All critical indexes are already in `install.sql`:
- `gas_tracker`: timestamp
- `token_transfers`: tx_hash, block_number, timestamp, addresses
- `dex_swaps`: tx_hash, block_number, timestamp, tokens
- `mev_opportunities`: type, status, detected_at, net_profit

### Query Optimization

- All SELECT queries use LIMIT
- Prepared statements prevent SQL injection
- Connection pooling via singleton pattern

---

## Security Features

✅ **API Key Protection**: Never exposed to frontend  
✅ **SQL Injection Prevention**: PDO prepared statements  
✅ **Input Validation**: All addresses and parameters validated  
✅ **Rate Limiting**: Enforced at application level  
✅ **CORS**: Restricted to app.poipoe.com  
✅ **Directory Protection**: .htaccess in logs/ and cache/  
✅ **Error Handling**: No internal errors exposed to users  

---

## Success Criteria

After 24 hours of operation, verify:

- ✅ All cron jobs running without errors
- ✅ Gas prices updated every minute
- ✅ At least 10 opportunities detected
- ✅ API endpoints respond in <200ms
- ✅ Zero database connection errors
- ✅ API usage < 90% of daily quota
- ✅ Error rate < 1% of total operations
- ✅ Frontend successfully fetches data

---

## Support & Contacts

**System Author**: MEV Pipeline System  
**Installation Date**: 2025-11-15  
**Version**: 1.0.0  

For issues or questions, check:
1. Log files in `logs/` directory
2. Error log table in database
3. Health check endpoint

---

## Next Steps

1. ✅ Complete installation steps above
2. ✅ Verify all cron jobs running
3. ✅ Monitor for 24 hours
4. ✅ Integrate with frontend
5. ⏳ Add more watched addresses (optional)
6. ⏳ Implement price oracle for accurate USD values (future)
7. ⏳ Add more DEX protocols (future)
8. ⏳ Implement ML-based opportunity scoring (future)

---

**Installation Complete! Your MEV Data Pipeline is now live.** 🚀
