<?php
/**
 * File: setup/test_installation.php
 * Purpose: Verify MEV pipeline installation and configuration
 * Usage: php setup/test_installation.php
 * Author: MEV Pipeline System
 * Last Modified: 2025-11-15
 */

echo "========================================\n";
echo "MEV Data Pipeline - Installation Test\n";
echo "========================================\n\n";

$errors = [];
$warnings = [];
$passed = 0;

// Test 1: PHP Version
echo "[1/10] Checking PHP version... ";
if (version_compare(PHP_VERSION, '7.4.0', '>=')) {
    echo "✓ PASS (PHP " . PHP_VERSION . ")\n";
    $passed++;
} else {
    echo "✗ FAIL (Need PHP 7.4+, found " . PHP_VERSION . ")\n";
    $errors[] = "PHP version too old";
}

// Test 2: Required Extensions
echo "[2/10] Checking PHP extensions... ";
$required = ['pdo', 'pdo_mysql', 'curl', 'json', 'mbstring'];
$missing = [];
foreach ($required as $ext) {
    if (!extension_loaded($ext)) {
        $missing[] = $ext;
    }
}
if (empty($missing)) {
    echo "✓ PASS\n";
    $passed++;
} else {
    echo "✗ FAIL (Missing: " . implode(', ', $missing) . ")\n";
    $errors[] = "Missing required PHP extensions";
}

// Test 3: Config Files
echo "[3/10] Checking configuration files... ";
$configFiles = [
    __DIR__ . '/../config/database.php',
    __DIR__ . '/../config/etherscan.php'
];
$missingConfigs = [];
foreach ($configFiles as $file) {
    if (!file_exists($file)) {
        $missingConfigs[] = basename($file);
    }
}
if (empty($missingConfigs)) {
    echo "✓ PASS\n";
    $passed++;
} else {
    echo "✗ FAIL (Missing: " . implode(', ', $missingConfigs) . ")\n";
    $errors[] = "Missing configuration files";
}

// Test 4: Database Connection
echo "[4/10] Testing database connection... ";
try {
    require_once __DIR__ . '/../config/database.php';
    $pdo = getDatabaseConnection();
    echo "✓ PASS\n";
    $passed++;
} catch (Exception $e) {
    echo "✗ FAIL (" . $e->getMessage() . ")\n";
    $errors[] = "Database connection failed";
    exit(1); // Cannot continue without database
}

// Test 5: Database Tables
echo "[5/10] Checking database tables... ";
$requiredTables = [
    'gas_tracker',
    'token_transfers',
    'dex_swaps',
    'contract_events',
    'mev_opportunities',
    'watched_addresses',
    'api_call_log',
    'system_config',
    'error_log'
];
$stmt = $pdo->query("SHOW TABLES");
$existingTables = $stmt->fetchAll(PDO::FETCH_COLUMN);
$missingTables = array_diff($requiredTables, $existingTables);

if (empty($missingTables)) {
    echo "✓ PASS (All 9 tables exist)\n";
    $passed++;
} else {
    echo "✗ FAIL (Missing: " . implode(', ', $missingTables) . ")\n";
    $errors[] = "Missing database tables - run install.sql";
}

// Test 6: Directory Permissions
echo "[6/10] Checking directory permissions... ";
$writableDirs = [
    __DIR__ . '/../logs',
    __DIR__ . '/../cache'
];
$permissionIssues = [];
foreach ($writableDirs as $dir) {
    if (!is_dir($dir)) {
        mkdir($dir, 0777, true);
    }
    if (!is_writable($dir)) {
        $permissionIssues[] = basename($dir);
    }
}
if (empty($permissionIssues)) {
    echo "✓ PASS\n";
    $passed++;
} else {
    echo "⚠ WARNING (Not writable: " . implode(', ', $permissionIssues) . ")\n";
    $warnings[] = "Some directories need write permissions (chmod 777)";
    $passed++; // Not critical, just warning
}

// Test 7: Helper Classes
echo "[7/10] Checking helper classes... ";
try {
    require_once __DIR__ . '/../helpers/Logger.php';
    require_once __DIR__ . '/../helpers/RateLimiter.php';
    require_once __DIR__ . '/../helpers/ApiClient.php';
    require_once __DIR__ . '/../helpers/Utils.php';
    echo "✓ PASS\n";
    $passed++;
} catch (Exception $e) {
    echo "✗ FAIL (" . $e->getMessage() . ")\n";
    $errors[] = "Helper classes not loading";
}

// Test 8: API Key Configuration
echo "[8/10] Checking Etherscan API key... ";
require_once __DIR__ . '/../config/etherscan.php';
if (defined('ETHERSCAN_API_KEY') && !empty(ETHERSCAN_API_KEY) && ETHERSCAN_API_KEY !== 'YOUR_KEY_HERE') {
    echo "✓ PASS\n";
    $passed++;
} else {
    echo "✗ FAIL (API key not configured)\n";
    $errors[] = "Etherscan API key not set";
}

// Test 9: Watched Addresses Seeded
echo "[9/10] Checking watched addresses... ";
$stmt = $pdo->query("SELECT COUNT(*) as count FROM watched_addresses");
$result = $stmt->fetch();
if ($result['count'] >= 5) {
    echo "✓ PASS ({$result['count']} addresses configured)\n";
    $passed++;
} else {
    echo "⚠ WARNING (Only {$result['count']} addresses, expected 6+)\n";
    $warnings[] = "Few watched addresses configured";
    $passed++; // Not critical
}

// Test 10: System Config Seeded
echo "[10/10] Checking system configuration... ";
$stmt = $pdo->query("SELECT COUNT(*) as count FROM system_config");
$result = $stmt->fetch();
if ($result['count'] >= 10) {
    echo "✓ PASS ({$result['count']} config entries)\n";
    $passed++;
} else {
    echo "⚠ WARNING (Only {$result['count']} config entries)\n";
    $warnings[] = "System configuration incomplete";
    $passed++; // Not critical
}

// Summary
echo "\n========================================\n";
echo "Test Summary\n";
echo "========================================\n";
echo "Passed: $passed/10\n";
echo "Errors: " . count($errors) . "\n";
echo "Warnings: " . count($warnings) . "\n\n";

if (!empty($errors)) {
    echo "❌ ERRORS:\n";
    foreach ($errors as $error) {
        echo "  - $error\n";
    }
    echo "\n";
}

if (!empty($warnings)) {
    echo "⚠️  WARNINGS:\n";
    foreach ($warnings as $warning) {
        echo "  - $warning\n";
    }
    echo "\n";
}

if (count($errors) === 0) {
    echo "✅ Installation Status: READY\n\n";
    echo "Next Steps:\n";
    echo "1. Verify cron jobs are configured in cPanel\n";
    echo "2. Run: php cron/cron_master.php --action=health\n";
    echo "3. Test API: https://core.poipoe.com/mevdata/api/\n";
    echo "4. Monitor: https://core.poipoe.com/mevdata/admin/\n";
    exit(0);
} else {
    echo "❌ Installation Status: FAILED\n";
    echo "Please fix the errors above before proceeding.\n";
    exit(1);
}
